"""大梁の結合状態指定.py

既存の物件データをベースに指定した大梁符号を両端“0：ピン”
とした物件データを作成します。
大梁符号名には以下のメタ文字（ワイルドカード）が使えます。
・$ → [A-Za-z]
・# → [0-9]

Copyright (C) 2026 UNION SYSTEM Inc.

"""

import tkinter as tk
from tkinter import messagebox
import os
import re
import csv
from Python import Ss7Python as cmd
import ss7_csv_input as csv_in
from typing import Tuple


def expand_wildcard_patterns(
    patterns: list, available_fugos: list
) -> Tuple[list, list]:
    """梁符号の入力文字列→該当する梁符号名

    Args:
        patterns (_type_): 梁符号の入力文字列
        available_fugos (_type_): 有効な梁符号名

    Returns:
        Tuple[list, list]: 該当する符号名, 該当しない符号名
    """
    matched = set()
    unmatched = []
    for pat in patterns:
        regex = re.escape(pat)
        regex = regex.replace(r"\$", r"[A-Za-z]").replace(r"\#", r"[0-9]")
        try:
            compiled = re.compile(f"^{regex}$", re.IGNORECASE)
        except re.error:
            unmatched.append(pat)
            continue
        match_found = False
        for fugou in available_fugos:
            if compiled.match(fugou):
                matched.add(fugou)
                match_found = True
        if not match_found:
            unmatched.append(pat)
        if unmatched:
            messagebox.showwarning(
                "一部未検出", f"以下の指定は該当なし:\n{', '.join(unmatched)}"
            )
    return list(matched), unmatched


def insert_ther_table_if_needed(csv_path: str) -> bool:
    """CSV入力データに梁の結合状態を挿入する

    Args:
        csv_path (str): CSV入力データのパス

    Returns:
        bool: True=成功 False=失敗
    """
    with open(csv_path, encoding="cp932", newline="") as f:
        rows = list(csv.reader(f))

    # 梁の結合状態の完全なヘッダ構成を定義
    ther_header = [
        ["name=梁の結合状態"],
        [
            "層",
            "フレーム-軸-軸",
            "ケース",
            "結合状態(鉛直面内)",
            "",
            "結合状態(水平面内)",
            "",
        ],
        ["", "", "", "左端", "右端", "左端", "右端"],
        ["<unit>", "", "", "", "", "", ""],
        ["", "", "", "kN*m/rad", "kN*m/rad", "kN*m/rad", "kN*m/rad"],
        ["<data>"],
    ]
    ther_table = ["name=梁の結合状態"]

    # ヘッダ全体がすでに存在するかを判定
    ther_table_exists = False
    for i in range(len(rows)):
        if rows[i] == ther_table:
            ther_table_exists = True
            break

    if not ther_table_exists:
        # 挿入位置を name=支点の状態の指定方法 の直前に決める
        insert_index = None
        for i, row in enumerate(rows):
            if row and row[0].strip() == "name=支点の状態の指定方法":
                insert_index = i
                break
        if insert_index is None:
            messagebox.showerror(
                "エラー", "'name=支点の状態の指定方法' が見つかりません。"
            )
            return False

        rows[insert_index:insert_index] = ther_header

        with open(csv_path, "w", encoding="cp932", newline="") as f:
            writer = csv.writer(f)
            writer.writerows(rows)

    return True


def append_joint_record_if_missing(csv_path: str, sou: str, ichi: str) -> bool:
    """新しい結合状態指定データを追加

    Args:
        csv_path (_type_): 入力CSVファイルのパス
        sou (_type_): 層
        ichi (_type_): 位置

    Returns:
        bool: True=成功  False=失敗
    """
    with open(csv_path, encoding="cp932", newline="") as f:
        rows = list(csv.reader(f))

    data_start_index = None
    for i, row in enumerate(rows):
        if row and row[0].strip() == "name=梁の結合状態":
            for j in range(i + 1, len(rows)):
                if rows[j] and rows[j][0].strip() == "<data>":
                    data_start_index = j + 1
                    break
            break

    if data_start_index is None:
        return False

    for row in rows[data_start_index:]:

        if len(row) > 1 and row[0].strip() == sou and row[1].strip() == ichi:
            return True  # 既に存在

    new_row = [sou, ichi, "標準", "0", "0", "-1", "-1", "<RE>"]

    rows.insert(data_start_index, new_row)

    with open(csv_path, "w", encoding="cp932", newline="") as f:
        writer = csv.writer(f)
        writer.writerows(rows)
    return True


def check_inputs(path: str, fugou_input: str, input_patterns: list) -> bool:
    """入力指定のチェック
    Args:
        path (str): 物件データのパス
        fugou_input (str): 符号名
        input_patterns (list): fugou_inputをカンマ区切りで分割したリスト
    Returns:
        bool: True:OK  False:NG
    """
    if not os.path.isdir(path):
        messagebox.showerror("エラー", "物件データのパスが存在しません。")
        return False
    if fugou_input == "":
        messagebox.showerror("エラー", "梁符号を入力してください。")
        return False
    if not input_patterns:
        messagebox.showerror("エラー", "有効な梁符号が入力されていません。")
        return False
    return True


def run_process() -> None:
    """[実行]ボタンのイベント関数"""
    # 入力内容の確認
    path = entry_path.get().strip()  # 物件データのパス
    fugou_input = entry_fugou.get().strip()  # ピン接にする梁符号
    input_patterns = [f.strip() for f in fugou_input.split(",") if f.strip()]
    if not check_inputs(path, fugou_input, input_patterns):
        return

    # 結合状態編集用入力CSVファイル
    inputcsv_path = os.path.join(path, "inp_all.csv")

    # 物件データを『SS7』で開く
    cmd.Init()
    cmd.Start("1.1.21.2", 1)
    dat = cmd.Open(path, 3, 3)
    if dat is None:
        messagebox.showerror("エラー", "物件データを開けませんでした。")
        cmd.End(2)
        return

    # 入力データを結合状態編集用入力CSVファイルに書き込む
    inp_dat = dat.GetInputData()
    if inp_dat is None:
        dat.Close(2)
        cmd.End(2)
        messagebox.showerror("エラー", "入力データが取得できませんでした。")
        return
    inp_dat.ExportInputCsv(inputcsv_path, 1, 1, 1)
    dat.Close(2)
    if not cmd.GetLastError().IsOK():
        messagebox.showerror("エラー", "CSV出力に失敗しました。")
        cmd.End(2)
        return

    # 大梁配置データ → 結合状態指定データ
    csvobj: csv_in.Ss7CsvInput = csv_in.Ss7CsvInput(inputcsv_path)
    table_layout: csv_in.Ss7CsvInputTable = csvobj.search_table("大梁配置")
    col_layer = table_layout.search_col1("層")
    col_axis = table_layout.search_col1("フレーム-軸-軸")
    col_fugou = table_layout.search_col1("符号")
    layout_fugos = [
        table_layout.get_data(row, col_fugou)
        for row in range(table_layout.get_row_count())
    ]
    matched_fugos, unmatched_patterns = expand_wildcard_patterns(
        input_patterns, layout_fugos
    )
    if not matched_fugos:
        messagebox.showinfo(
            "確認", "該当する梁符号が見当たりません。処理を中断します。"
        )
        cmd.End(2)
        return

    target_keys = []
    for row in range(table_layout.get_row_count()):
        fugou = table_layout.get_data(row, col_fugou)
        if fugou in matched_fugos:
            sou = table_layout.get_data(row, col_layer)
            ichi = table_layout.get_data(row, col_axis)
            target_keys.append((sou, ichi))

    insert_ther_table_if_needed(inputcsv_path)
    csvobj = csv_in.Ss7CsvInput(inputcsv_path)
    table_joint: csv_in.Ss7CsvInputTable = csvobj.search_table("梁の結合状態")
    col_vleft = table_joint.search_col2("結合状態(鉛直面内)", "左端")
    col_vright = table_joint.search_col2("結合状態(鉛直面内)", "右端")

    for sou, ichi in target_keys:
        found = -1
        for row in range(table_joint.get_row_count()):
            if (table_joint.get_data(row, 0) == sou) and (
                table_joint.get_data(row, 1) == ichi
            ):
                found = row
                break

        if found == -1:
            append_joint_record_if_missing(inputcsv_path, sou, ichi)
            csvobj = csv_in.Ss7CsvInput(inputcsv_path)
            table_joint = csvobj.search_table("梁の結合状態")
            col_vleft = table_joint.search_col2("結合状態(鉛直面内)", "左端")
            col_vright = table_joint.search_col2("結合状態(鉛直面内)", "右端")

        if table_joint:
            table_joint.set_data_key2(sou, ichi, col_vleft, "0")
            table_joint.set_data_key2(sou, ichi, col_vright, "0")
        else:
            messagebox.showerror(
                "エラー", "梁の結合状態テーブルが見つかりませんでした。"
            )
            cmd.End(2)
            return

    csvobj.write(inputcsv_path)
    base, ext = os.path.splitext(path)
    output_path = base + "_pin" + ext
    cmd.CreateDataCsv(inputcsv_path, output_path, 1)

    if cmd.GetLastError().IsOK():
        messagebox.showinfo(
            "成功", f"新しい物件データが作成されました。\n{output_path}"
        )
    else:
        messagebox.showerror("エラー", "物件データの作成に失敗しました。")
    cmd.End(2)


def close_window() -> None:
    """[終了]ボタンのイベント関数"""
    root.destroy()


# メイン
root = tk.Tk()
root.title("梁の結合状態変更データ作成")

tk.Label(root, text="物件データのパス:").grid(row=0, column=0, sticky="e")
entry_path = tk.Entry(root, width=50)
entry_path.grid(row=0, column=1, padx=5, pady=5)

tk.Label(root, text="梁符号:").grid(row=1, column=0, sticky="e")
entry_fugou = tk.Entry(root, width=50)
entry_fugou.grid(row=1, column=1, padx=5, pady=5)

tk.Button(root, text="実行(A)", command=run_process).grid(
    row=2, column=0, padx=5, pady=10
)
tk.Button(root, text="終了(X)", command=close_window).grid(
    row=2, column=1, padx=5, pady=10, sticky="w"
)

root.mainloop()
