"""
本プログラムは、円形鋼管の曲げを考慮した組合せ応力度をExcelで検討するプログラムです。<br>

曲げモーメントを考慮した組合せ応力度を検討したい円形鋼管の柱符号（複数指定可）と
計算結果シートの「S柱断面算定表.csv」と「柱設計応力表.csv」を指定することで、
S柱断面算定表の採用応力値の並びに、採用位置とケースに対応した直交方向の曲げモーメントの値を追記して、
新たな「S柱断面算定表_追加済.csv」を出力します（材料強度と許容圧縮応力度を文字列と数値に分けます）。
さらに、Excelの関数を用いて直交方向の曲げを考慮した応力度と組み合わせ応力度を計算するExcelブックを
出力します。

複数の梁符号をカンマで区切って指定できます。
「柱設計応力表.csv」は、地震時と長期を加えて、（長期に地震時を加えてもOK）
一つのファイルにしてください。このとき、ApNameや計算日時など重複していても問題ありません。
二つのファイルを開いて、どちらか一方のすべてを選択＆コピーして、もう一方にペーストしてください。

Copyright (C) 2026 UNION SYSTEM Inc.

"""

import csv
import re
from typing import Optional
import ss7_csv_result as csv_result
from openpyxl import Workbook
from openpyxl.styles import Alignment


def get_nearest_f_values(row_idx: int) -> Optional[int]:
    """
    ワークシートwsの中から["鉄骨", "柱頭", "Ｆ値", "柱脚", "Ｆ値"]を含む行番号を返す

    Args:
        row_idx (int): 探す範囲の最大行番号

    Returns:
        int: 鉄骨F値の行番号を探す
    """
    for r in range(row_idx - 1, 0, -1):
        if [ws.cell(row=r, column=c).value for c in range(1, 6)] == [
            "鉄骨",
            "柱頭",
            "Ｆ値",
            "柱脚",
            "Ｆ値",
        ]:
            Festigkeit_row = r + 1
            return Festigkeit_row
    return None


# ---------- 設定 ----------
symbol_target: str = "PC2,2C2,2C3"
symbol_targets: list[str] = [s.strip() for s in symbol_target.split(",")]
s_csv_path: str = r"C:\enkei_kumiawase\S柱断面算定表.csv"
result_csv_path: str = r"C:\enkei_kumiawase\S柱断面算定表_追加済.csv"
stress_csv_path: str = r"C:\enkei_kumiawase\設計応力表.csv"
Excel_path: str = r"C:\enkei_kumiawase\S柱断面算定表_組み合わせ.xlsx"

# ---------- ヘッダーとパターン定義 ----------
target_header: str = "     鉄　骨   　   柱頭　     Ｆ値　 　 柱脚　     Ｆ値"
material_pattern: str = r"\s*\[(.*?)\s*\]\s*([\d\.]+)\s*\[\s*(.*?)\s*\]\s*([\d\.]+)"
Allowable_pattern: str = r"(.*?)\s*([\d\.]+)"
pattern: re.Pattern = re.compile(Allowable_pattern)

# ---------- SS7設計応力表の読み込み ----------
csvss7: csv_result.Ss7CsvResult = csv_result.Ss7CsvResult(stress_csv_path)

# ---------- S柱断面算定表の読み込み ----------
with open(s_csv_path, encoding="cp932") as f:
    reader = csv.reader(f)
    rows = list(reader)

output_rows: list[list[str]] = []
i: int = 0

while i < len(rows):
    row = rows[i]

    if row and row[0].strip() == target_header.strip():
        output_rows.append(["鉄骨", "柱頭", "Ｆ値", "柱脚", "Ｆ値"])
        if i + 1 < len(rows):
            next_row = rows[i + 1]
            if next_row:
                match = re.match(material_pattern, next_row[0])
                if match:
                    mat1, val1, mat2, val2 = match.groups()
                    output_rows.append(["", mat1, val1, mat2, val2])
                else:
                    output_rows.append(["不一致", next_row[0], "", ""])
            else:
                output_rows.append(["空行", "", "", ""])
        i += 2
        continue

    if row and row[0].startswith("[") and row[0][1:-1] in symbol_targets:
        symbol = row[0][1:-1]
        floor = row[2][1:] if len(row) > 2 and row[2].startswith("[") else ""
        x_axis = row[4] if len(row) > 4 else ""
        y_axis = row[6].rstrip("]") if len(row) > 6 else ""

        case_col = -1
        for idx, val in enumerate(row):
            if "ｹｰｽ" in val:
                case_col = idx
                break

        if case_col == -1:
            output_rows.append(row)
            i += 1
            continue

        while len(row) <= 19:
            row.append("")
        row[19] = "直交M"
        output_rows.append(row)

        for j in range(1, 7):
            if i + j >= len(rows):
                break
            data_row = rows[i + j]
            case = data_row[case_col].strip() if case_col < len(data_row) else ""
            moment = ""
            table: csv_result.Ss7CsvResultTable
            if case == "L":
                table = csvss7.search_table_case("柱設計応力表", "長期", "")
            else:
                table = csvss7.search_table_case("柱設計応力表", "地震時", "")
            try:
                if data_row[9] == "<X>柱頭":
                    col = table.search_col2("曲げy", "柱頭")
                elif j <= 2 and data_row[9] == "中央":
                    col = table.search_col2("曲げy", "中央")
                elif j <= 3 and data_row[9] == "柱脚":
                    col = table.search_col2("曲げy", "柱脚")
                elif data_row[9] == "<Y>柱頭":
                    col = table.search_col2("曲げx", "柱頭")
                elif j >= 4 and data_row[9] == "中央":
                    col = table.search_col2("曲げx", "中央")
                elif j >= 4 and data_row[9] == "柱脚":
                    col = table.search_col2("曲げx", "柱脚")
                else:
                    col = None
                if col is not None:
                    moment = table.get_data_key5(
                        floor, x_axis, y_axis, symbol, case, col
                    )
            except Exception:
                moment = ""

            while len(data_row) <= 19:
                data_row.append("")
            if moment:
                data_row[19] = moment
            output_rows.append(data_row)

        i += j + 1
        continue

    if len(row) >= 2:
        col1 = row[0].strip()
        col2 = row[1].strip()
        if col1 == "":
            match = pattern.match(col2)
            if match:
                row[0] = match.group(1).strip()
                row[1] = match.group(2).strip()

    output_rows.append(row)
    i += 1

# ---------- CSV書き出し ----------
with open(result_csv_path, "w", newline="", encoding="cp932") as f:
    writer = csv.writer(f)
    writer.writerows(output_rows)

print("CSV出力完了：", result_csv_path)

# ---------- Excel書き出し ----------
wb = Workbook()
ws = wb.active
ws.title = "S柱断面"
Dangposi_count = 1

for r_idx, row in enumerate(output_rows, start=1):
    for c_idx, val in enumerate(row, start=1):
        ws.cell(row=r_idx, column=c_idx, value=val)

for i in range(1, ws.max_row + 1):
    if ws.cell(row=i, column=20).value == "直交M":
        i_row = i
        ws.cell(row=i_row, column=22).value = "√MX2+MY2"
        for j in range(7):
            data_row_combination_table: int = i_row + 1 + j
            Dangposi_count = j
            if ws.cell(row=data_row_combination_table, column=20).value == "組合せ":
                for offset, header in enumerate(
                    ["σc", "σb", "τ", "組合せL（σb考慮）", "組合せS（σb考慮）"]
                ):
                    ws.cell(
                        row=data_row_combination_table, column=22 + offset
                    ).value = header
                break
            formula = f"=SQRT(R{data_row_combination_table}^2+\
                    T{data_row_combination_table}^2)"
            ws.cell(row=data_row_combination_table, column=22).value = formula

    if ws.cell(row=i, column=22).value == "σc":
        i_row = i
        Festigkeit_row = get_nearest_f_values(i)
        for j in range(Dangposi_count):
            row_F_table = i_row + 1 + j
            row_m = i_row + j - Dangposi_count
            if not ws.cell(row=row_F_table, column=10).value:
                break
            formula_sigma_c = f"=ABS(Q{row_m}/L{row_F_table}*10)"
            ws.cell(row=row_F_table, column=22).value = formula_sigma_c
            formula_sigma_b = f"=ABS(V{row_m}/K{row_F_table}*1000)"
            ws.cell(row=row_F_table, column=23).value = formula_sigma_b
            formula_tau = f'=IF(S{row_m}<>0,S{row_m}/M{row_F_table}*10,"-")'
            ws.cell(row=row_F_table, column=24).value = formula_tau
            judge = ws.cell(row=row_F_table, column=10).value
            use_f = "C" if "柱頭" in judge or "中央" in judge else "E"
            formula_comb_L = f'=IF(S{row_m}<>0,IF(P{row_m}="L",\
                    (SQRT(((V{row_F_table}+\
                    W{row_F_table})^2)+\
                    (3*(X{row_F_table}^2)))/\
                    {use_f}{Festigkeit_row}*1.5),"-"),"-")'
            formula_comb_S = f'=IF(S{row_m}<>0,IF(P{row_m}="L","-",\
                    (SQRT(((V{row_F_table}+\
                    W{row_F_table})^2)+\
                    (3*(X{row_F_table}^2)))/\
                    {use_f}{Festigkeit_row})),"-")'
            ws.cell(row=row_F_table, column=25).value = formula_comb_L
            ws.cell(row=row_F_table, column=26).value = formula_comb_S


# 小数第1位の書式設定
float_columns = list(range(22, 27))  # V～Z列
for row in ws.iter_rows(min_row=1, max_row=ws.max_row):
    for col_idx in float_columns:
        cell = row[col_idx - 1]
        if isinstance(cell.value, (int, float)) or (
            isinstance(cell.value, str) and cell.value.startswith("=")
        ):
            cell.number_format = "0.0"


# セルの整列（オプション）
for row in ws.iter_rows():
    for cell in row:
        cell.alignment = Alignment(vertical="center")

wb.save(Excel_path)
print("Excel出力完了：", Excel_path)
