#!/usr/bin/python
# -*- coding: utf-8 -*-
"""SS7から出力された結果CSVファイルを扱う
以下のクラスを含みます
CSVファイルクラス：Ss7CsvResult
テーブルクラス：Ss7CsvResultTable
"""
import csv
from typing import List, Optional


def cut_str(text: str, cut: str) -> str:
    """
    文字列カット関数
    Args:
        text (str): カットしたい文字を含む文字列
        cut (str): カットする文字列
    Returns:
        str: カット後の文字列
    """
    pos = text.find(cut)
    if 0 <= pos:
        return text[pos + len(cut) : len(text)]
    return text

def is_match_key( rec:list[str], keys:list[str], keys_pre:list[str], size:int ) -> bool:
    """
    Keyの一致判定
    Args:
        keys:list[str]: キー文字列
        keys_pre:list[str]: 前行のキー文字列
        size (int): キーの数
    Returns:
        bool: 判定結果 true:一致 false:不一致
    """
    ret:bool = True
    for i in range(size):
        # 値がNULLなら前行の値を採用
        key_i:str = rec[i]
        if key_i == "":
            key_i = keys_pre[i]

        # 判定
        if key_i != keys[i]:
            ret = False

        # 今回の値を前行の値として保存
        keys_pre[i] = key_i

    return ret

class Ss7CsvResultTable:
    """
    テーブルクラス
    """

    def __init__(self) -> None:
        """
        コンストラクタ
        """
        self.name: str = ""
        self.case: str = ""
        self.tab: str = ""
        self.fields: List[List[str]] = []  # 2次元配列の宣言がわからない
        self.records: List[List[str]] = []  # 2次元配列の宣言がわからない

    def set_name(self, name: str, case: str, tab: str) -> None:
        """
        テーブル名の登録
        Args:
            name (str): テーブル名（name=）
            case (str): ケース名（case=）
            tab (str): タブ名（tab=）
        """
        self.name = name
        self.case = case
        self.tab = tab

    def set_field(self, data: [str]) -> None:
        """
        列名の登録（単位も含まれます）
        Args:
            data (str]): CSV一行分のデータ
        """
        self.fields.append(data)

    def set_data(self, data: [str]) -> None:
        """
        値の登録
        Args:
            data (str]): CSV一行分のデータ
        """
        self.records.append(data)

    def get_row_count(self) -> int:
        """
        データの行数の取得
        Returns:
            int: 行数
        """
        return len(self.records)

    def get_col_count(self) -> int:
        """
        データの列数の取得
        Returns:
            int: 列数
        """
        return len(self.records[0])

    def get(self, row: int, col: int) -> str:
        """
        値の取得
        Args:
            row (int): 行インデックス（0～）
            col (int): 列インデックス（0～）
        Returns:
            str: 値
        """
        if row < len(self.records):
            if col < len(self.records[row]):
                return self.records[row][col]
        return ""

    def get_key_row1(self, key: str, col: int) -> str:
        """
        値の取得
        Args:
            key (str): キー文字列
            col (int): 列インデックス（0～）
        Returns:
            str: キーで検索した値
        """
        row:int = 0
        if col < self.get_col_count():
            for rec in self.records:
                if rec[0] == key:
                    return row
                row = row + 1
        return -1

    def get_key_row2(self, key1: str, key2: str, col: int) -> str:
        """
        値の取得
        Args:
            key (str): キー文字列
            col (int): 列インデックス（0～）
        Returns:
            str: キーで検索した値
        """
        keys:list[str] = [key1, key2]
        keys_pre:list[str] = ["",""]
        row:int = 0
        if col < self.get_col_count():
            for rec in self.records:
                if is_match_key( rec, keys, keys_pre, 2 ) == True:
                    return row
                row = row + 1
        return -1
    
    def get_key_row3(self, key1: str, key2: str, key3: str, col: int) -> str:
        """
        値の取得
        Args:
            key (str): キー文字列
            col (int): 列インデックス（0～）
        Returns:
            str: キーで検索した値
        """
        keys:list[str] = [key1, key2, key3]
        keys_pre:list[str] = ["","",""]
        row:int = 0
        if col < self.get_col_count():
            for rec in self.records:
                if is_match_key( rec, keys, keys_pre, 3 ) == True:
                    return row
                row = row + 1
        return -1

    def get_data_key1(self, key: str, col: int) -> str:
        """
        値の取得
        Args:
            key (str): キー文字列
            col (int): 列インデックス（0～）
        Returns:
            str: キーで検索した値
        """
        if col < self.get_col_count():
            for rec in self.records:
                if rec[0] == key:
                    if col < len(rec):
                        return rec[col]
        return ""

    def get_data_key2(self, key1: str, key2: str, col: int) -> str:
        """
        値の取得
        Args:
            key1～ (str): キー文字列
            col (int): 列インデックス（0～）
        Returns:
            str: キーで検索した値
        """
        keys:list[str] = [key1, key2]
        keys_pre:list[str] = ["",""]
        if col < self.get_col_count():
            for rec in self.records:
                if is_match_key( rec, keys, keys_pre, 2 ) == True:
                    return rec[col]
        return ""

    def get_data_key3(self, key1: str, key2: str, key3: str, col: int) -> str:
        """
        値の取得
        Args:
            key1～ (str): キー文字列
            col (int): 列インデックス（0～）
        Returns:
            str: キーで検索した値
        """
        keys:list[str] = [key1, key2, key3]
        keys_pre:list[str] = ["","",""]
        if col < self.get_col_count():
            for rec in self.records:
                if is_match_key( rec, keys, keys_pre, 3 ) == True:
                    return rec[col]
        return ""

    def get_data_key4(
        self, key1: str, key2: str, key3: str, key4: str, col: int
    ) -> str:
        """
        値の取得
        Args:
            key1～ (str): キー文字列
            col (int): 列インデックス（0～）
        Returns:
            str: キーで検索した値
        """
        keys:list[str] = [key1, key2, key3, key4]
        keys_pre:list[str] = ["","","",""]
        if col < self.get_col_count():
            for rec in self.records:
                if is_match_key( rec, keys, keys_pre, 4 ) == True:
                    return rec[col]
        return ""

    def get_data_key5(
        self, key1: str, key2: str, key3: str, key4: str, key5: str, col: int
    ) -> str:
        """
        値の取得
        Args:
            key1～ (str): キー文字列
            col (int): 列インデックス（0～）
        Returns:
            str: キーで検索した値
        """
        keys:list[str] = [key1, key2, key3, key4, key5]
        keys_pre:list[str] = ["","","","",""]
        if col < self.get_col_count():
            for rec in self.records:
                if is_match_key( rec, keys, keys_pre, 5 ) == True:
                    return rec[col]
        return ""

    def search_col1(self, field: str) -> int:
        """
        列の検索
        Args:
            field (str): 列名
        Returns:
            int: 列インデックス（0～）
        """
        col = 0
        if 1 <= len(self.fields):
            for self_field in self.fields[0]:
                if self_field == field:
                    return col
                col = col + 1
        return -1

    def search_col2(self, field1: str, field2: str) -> int:
        """
        列の検索
        Args:
            field1 (str): 列名（上段）
            field2 (str): 列名（下段）
        Returns:
            int: 列インデックス（0～）
        """
        col = 0
        self_field1_pre = ""
        if 2 <= len(self.fields):
            for field in zip(self.fields[0], self.fields[1]):
                self_field1 = field[0]
                self_field2 = field[1]

                # 上段列名が空白の場合は直前の列名を引き継ぐ
                if self_field1 == "":
                    self_field1 = self_field1_pre

                if self_field1 == field1 and self_field2 == field2:
                    return col

                col = col + 1
                self_field1_pre = self_field1
        return -1

    def search_col3(self, field1: str, field2: str, field3: str) -> int:
        """
        列の検索
        Args:
            field1 (str): 列名（上段）
            field2 (str): 列名（中段）
            field3 (str): 列名（下段）
        Returns:
            int: 列インデックス（0～）
        """
        col = 0
        self_field1_pre = ""
        self_field2_pre = ""
        if 3 <= len(self.fields):
            for field in zip(self.fields[0], self.fields[1], self.fields[2]):
                self_field1 = field[0]
                self_field2 = field[1]
                self_field3 = field[2]

                # 上段列名が空白の場合は直前の列名を引き継ぐ
                if self_field1 == "":
                    self_field1 = self_field1_pre
                if self_field2 == "":
                    self_field2 = self_field2_pre

                if (
                    self_field1 == field1
                    and self_field2 == field2
                    and self_field3 == field3
                ):
                    return col

                col = col + 1
                self_field1_pre = self_field1
                self_field2_pre = self_field2
        return -1


class Ss7CsvResult:
    """
    結果CSV読み込みクラス
    """

    def __init__(self, file_path: str) -> None:
        """
        コンストラクタ
        Args:
            file_path (str): CSVファイルのパス
        """
        self.path = file_path
        self.headers = []
        self.tables = []

        with open(file_path, mode="r", encoding="cp932") as File:
            self.reader = csv.reader(File)
            bHeader = True
            bTable = False
            bData = False
            for row_data in self.reader:
                if 0 < len(row_data):
                    if 0 <= row_data[0].find("name="):
                        bHeader = False
                        bTable = True
                        name = cut_str(row_data[0], "name=")
                        case = ""
                        if 1 < len(row_data) and 0 <= row_data[1].find("case="):
                            case = cut_str(row_data[1], "case=")
                        elif 2 < len(row_data) and 0 <= row_data[2].find("case="):
                            case = cut_str(row_data[2], "case=")

                        tab = ""
                        if 1 < len(row_data) and 0 <= row_data[1].find("tab="):
                            tab = cut_str(row_data[1], "tab=")
                        if 2 < len(row_data) and 0 <= row_data[2].find("tab="):
                            tab = cut_str(row_data[2], "tab=")

                        self.tables.append(Ss7CsvResultTable())
                        table = self.tables[-1]
                        table.set_name(name, case, tab)
                        continue

                    elif len(row_data) == 1 and 0 <= row_data[0].find("<data>"):
                        bData = True
                        continue

                    if bTable:
                        if bData:
                            table.set_data(row_data)
                        else:
                            table.set_field(row_data)

                    if bHeader:
                        self.headers.append(row_data)

                else:
                    bTable = False
                    bData = False

    def get_all_tables(self) -> [Ss7CsvResultTable]:
        """
        全テーブルの取得
        Returns:
            [Ss7CsvResultTable]: テーブルリスト
        """
        return self.tables

    def search_table(self, table_name: str) -> Optional[Ss7CsvResultTable]:
        """
        テーブルの検索
        Args:
            table_name (str): テーブル名（name=）
        Returns:
            Optional[Ss7CsvResultTable]: テーブル（見つからなかった場合はNone）
        """
        for table in self.tables:
            if table.name == table_name:
                return table
        return None

    def search_table_case(
        self, table_name: str, case_name: str, tab_name: str
    ) -> Optional[Ss7CsvResultTable]:
        """
        テーブルの検索
        Args:
            table_name (str): テーブル名（name=）
            case_name (str): ケース名（case=）
            tab_name (str): タブ名（tab=）
        Returns:
            Optional[Ss7CsvInputTable]: テーブル（見つからなかった場合はNone）
        """
        for table in self.tables:
            if table.name == table_name:
                if table.case == case_name:
                    if table.tab == tab_name:
                        return table
        return None
