"""耐震診断の終局強度とF指標の計算

RC長方形柱に対する曲げ終局強度は、耐震診断で用いられるピース断面分割により計算します。
せん断終局強度は、RC造耐震診断基準の“付則1.1 (3)せん断終局強度の算定”により計算します。
また、RC長方形柱のF指標は、RC造耐震診断基準の“3.2.3 靭性指標”、“付則1.2 柱の終局変形”および
“付則1.3 柱の曲げ降伏変形”により計算します。

Copyright (C) 2026 UNION SYSTEM Inc.

"""

from Python import Ss7Python as Cmd
from typing import Tuple
import os
import csv
import re
import math


def del_csv(resultcsv_path: str) -> None:
    """既存の結果csvの削除

    既存の結果csvを削除します。

    Args:
        resultcsv_path (str): 結果csvのpath

    """

    # 既存の結果csvを削除する
    if os.path.isfile(resultcsv_path):
        os.remove(resultcsv_path)


def search_csv(list_text: list) -> Tuple[int, int, int, int, int, int, int, int]:
    """結果CSV出力から出力項目ごとのデータ読み込みの開始行数と終了行数の取得 (その1)

    以下の出力項目のデータ読み込みの開始行数と終了行数を取得する
        ・コンクリート材料の開始行数
        ・鉄筋材料の開始行数
        ・鉄筋径と使用範囲の開始行数
        ・柱部材断面情報の開始行数
        ・コンクリート材料の終了行数
        ・鉄筋材料の終了行数
        ・鉄筋径と使用範囲の終了行数
        ・柱部材断面情報の終了行数

    Args:
        list_text (list): 結果CSV出力のリスト
    Returns:
        int: [name=コンクリート材料] の開始行数
        int: [name=鉄筋材料] の開始行数
        int: [name=鉄筋径と使用範囲] の開始行数
        int: [name=柱部材断面情報] の開始行数
        int: [name=コンクリート材料] の終了行数
        int: [name=鉄筋材料] の終了行数
        int: [name=鉄筋径と使用範囲] の終了行数
        int: [name=柱部材断面情報] の終了行数

    """

    search_t1: str = "コンクリート材料"
    search_t2: str = "鉄筋材料"
    search_t3: str = "鉄筋径と使用範囲"
    search_t4: str = "柱部材断面情報"

    i_conc_st: int = 0  # [name=コンクリート材料] の開始行数
    i_tekin_st: int = 0  # [name=鉄筋材料] の開始行数
    i_tekinkei_st: int = 0  # [name=鉄筋径と使用範囲] の開始行数
    i_colm_st: int = 0  # [name=柱部材断面情報] の開始行数
    i_conc_ed: int = 0  # [name=コンクリート材料] の終了行数
    i_tekin_ed: int = 0  # [name=鉄筋材料] の終了行数
    i_tekinkei_ed: int = 0  # [name=鉄筋径と使用範囲] の終了行数
    i_colm_ed: int = 0  # [name=柱部材断面情報] の終了行数

    # 結果CSV出力から出力項目ごとの出力開始行数と終了行数の取得
    i_rw: int = -1
    for item in list_text:
        i_rw += 1
        for key in item:
            # name=コンクリート材料の行数
            if search_t1 in key:
                i_conc_st = i_rw  # 開始行
                break

            # name=鉄筋材料の行数
            if search_t2 in key:
                i_conc_ed = i_rw - 1  # 終了行
                i_tekin_st = i_rw  # 開始行
                break

            # name=鉄筋径と使用範囲の行数
            if search_t3 in key:
                i_tekin_ed = i_rw - 1  # 終了行
                i_tekinkei_st = i_rw  # 開始行
                break

            # name=柱部材断面情報の行数
            if search_t4 in key:
                i_tekinkei_ed = i_rw - 1  # 終了行
                i_colm_st = i_rw  # 開始行
                break

    # name=柱部材断面情報の行数
    i_colm_ed = len(list_text)  # 終了行

    return (
        i_conc_st,
        i_tekin_st,
        i_tekinkei_st,
        i_colm_st,
        i_conc_ed,
        i_tekin_ed,
        i_tekinkei_ed,
        i_colm_ed,
    )


def search_csv2(list_text: list, i_conc_st: int, i_conc_ed: int) -> int:
    """結果CSV出力から出力項目ごとの<data>の行数の取得 (その2)

    以下の出力項目の開始行数を取得する
    ・コンクリート材料の<data>の行数

    Args:
        list_text (list): 結果CSV出力のリスト
        i_conc_st (list): [name=コンクリート材料] の開始行数
        i_conc_ed (list): [name=コンクリート材料] の終了行数
    Returns:
        int: コンクリート材料の<data>の行数

    """

    # 結果CSV出力から出力項目ごとの<data>の出力開始行数の取得
    # コンクリート材料の<data>の行数
    search_t: str = "<data>"

    for i in range(i_conc_st, i_conc_ed):
        for key in list_text[i]:
            if search_t in key:
                i_conc_st = i  # 開始行
                break

    return i_conc_st


def search_csv3(list_text: list, i_tekin_st: int, i_tekin_ed: int) -> int:
    """結果CSV出力から出力項目ごとの<data>の行数の取得 (その3)

    以下の出力項目の開始行数を取得する
    ・鉄筋材料の<data>の行数

    Args:
        list_text (list): 結果CSV出力のリスト
        i_tekin_st (int): [name=鉄筋材料] の開始行数
        i_tekin_ed (int): [name=鉄筋材料] の終了行数
    Returns:
        int: 鉄筋材料の<data>の行数

    """

    # 鉄筋材料の<data>の行数
    search_t: str = "<data>"

    for i in range(i_tekin_st, i_tekin_ed):
        for key in list_text[i]:
            if search_t in key:
                i_tekin_st = i  # 開始行
                break

    return i_tekin_st


def search_csv4(list_text: list, i_tekinkei_st: int, i_tekinkei_ed: int) -> int:
    """結果CSV出力から出力項目ごとの<data>の行数の取得 (その4)

    以下の出力項目の開始行数を取得する
    ・鉄筋径と使用範囲の<data>の行数

    Args:
        list_text (list): 結果CSV出力のリスト
        i_tekinkei_st (int): [name=鉄筋径と使用範囲] の開始行数
        i_tekinkei_ed (int): [name=鉄筋径と使用範囲] の終了行数
    Returns:
        int: 鉄筋径と使用範囲の<data>の行数

    """

    # 鉄筋径と使用範囲の<data>の行数
    search_t: str = "<data>"

    for i in range(i_tekinkei_st, i_tekinkei_ed):
        for key in list_text[i]:
            if search_t in key:
                i_tekinkei_st = i  # 開始行
                break

    return i_tekinkei_st


def search_csv5(list_text: list, i_colm_st: int, i_colm_ed: int) -> int:
    """結果CSV出力から出力項目ごとの<data>の行数の取得 (その5)

    以下の出力項目の開始行数を取得する
    ・柱部材断面情報の<data>の行数

    Args:
        list_text (list): 結果CSV出力のリスト
        i_colm_st (int): [name=柱部材断面情報] の開始行数
        i_colm_ed (int): [name=柱部材断面情報] の終了行数
    Returns:
        int: 柱部材断面情報の<data>の行数

    """

    # 柱部材断面情報の<data>の行数
    search_t: str = "<data>"

    for i in range(i_colm_st, i_colm_ed):
        for key in list_text[i]:
            if search_t in key:
                i_colm_st = i  # 開始行
                break

    return i_colm_st


def get_result_csv2(list_text: list, i_conc_st: int, i_conc_ed: int) -> list:
    """コンクリート材料のデータ抽出

    結果CSV出力のリストからコンクリート材料のデータを抽出します。

    Args:
        list_text (list): 結果CSV出力のリスト
        i_conc_st (int): コンクリート材料の<data>の行数
        i_conc_ed (int): [name=コンクリート材料] の終了行数
    Returns:
        list: コンクリート材料のデータリスト

    """

    # 出力項目それぞれにリストを用意し、データを格納する
    # コンクリート材料のデータ
    for i in range(i_conc_st, i_conc_ed):
        if i + 1 < i_conc_ed:
            if len(list_text[i + 1]) != 0:
                list_conc.append(list_text[i + 1])

    return list_conc


def get_result_csv3(list_text: list, i_tekin_st: int, i_tekin_ed: int) -> list:
    """鉄筋材料のデータ抽出

    結果CSV出力のリストから鉄筋材料のデータを抽出します。

    Args:
        list_text (list): 結果CSV出力のリスト
        i_tekin_st (int): 鉄筋材料の<data>の行数
        i_tekin_ed (int): [name=鉄筋材料] の終了行数
    Returns:
        list: 鉄筋材料のデータリスト

    """

    # 出力項目それぞれにリストを用意し、データを格納する
    # 鉄筋材料のデータ
    for i in range(i_tekin_st, i_tekin_ed):
        if i + 1 < i_tekin_ed:
            if len(list_text[i + 1]) != 0:
                list_tekin.append(list_text[i + 1])

    return list_tekin


def get_result_csv4(list_text: list, i_tekinkei_st: int, i_tekinkei_ed: int) -> list:
    """鉄筋径と使用範囲のデータ抽出

    結果CSV出力のリストから鉄筋径と使用範囲のデータを抽出します。

    Args:
        list_text (list): 結果CSV出力のリスト
        i_tekinkei_st (int): 鉄筋径と使用範囲の<data>の行数
        i_tekinkei_ed (int): [name=鉄筋径と使用範囲] の終了行数
    Returns:
        list: 鉄筋径と使用範囲のデータリスト

    """

    # 出力項目それぞれにリストを用意し、データを格納する
    # 鉄筋径と使用範囲のデータ
    for i in range(i_tekinkei_st, i_tekinkei_ed):
        if i + 1 < i_tekinkei_ed:
            if len(list_text[i + 1]) != 0:
                list_tekinkei.append(list_text[i + 1])

    return list_tekinkei


def get_result_csv5(list_text: list, i_colm_st: int, i_colm_ed: int) -> list:
    """柱部材断面情報のデータ抽出

    結果CSV出力のリストから柱部材断面情報のデータを抽出します。

    Args:
        list_text (list): 結果CSV出力のリスト
        i_colm_st (int): 柱部材断面情報の<data>の行数
        i_colm_ed (int): [name=柱部材断面情報] の終了行数
    Returns:
        list: 柱部材断面情報のデータリスト

    """

    # 出力項目それぞれにリストを用意し、データを格納する
    # 柱部材断面情報のデータ
    for i in range(i_colm_st, i_colm_ed):
        if i + 1 < i_colm_ed:
            if len(list_text[i + 1]) != 0:
                list_colm.append(list_text[i + 1])

    return list_colm


def get_atc(data_c_atc: str) -> Tuple[
    float,
    float,
    str,
]:
    """柱の主筋本数、主筋断面積 および 径のデータ取得

    柱部材断面情報の主筋データから柱の主筋本数、主筋断面積 および 径を取得します。

    Args:
        data_c_atc (str): 柱部材断面情報の主筋データ
    Returns:
        float: 鉄筋本数
        float: 主筋断面積
        str: 鉄筋径

    """

    # 柱の主筋本数、主筋断面積 および 径のデータ取得
    d_hon: float = 0.0  # 鉄筋本数
    d_at: float = 0.0  # 主筋断面積
    str_kei: str = " "  # 鉄筋径

    # 主筋本数-径
    if "-" in data_c_atc:
        # 主筋本数-径:##-D## の処理
        im = data_c_atc.index("-")  # 文字列位置

        # 寄筋の指定
        if "/" in data_c_atc:
            # 主筋本数-径:##/2-D## の処理
            im2 = data_c_atc[:im].index("/")  # 文字列位置

            if "(" in data_c_atc:
                # (主筋断面積) 径:(####)/2-D## の処理
                im3 = data_c_atc[:im2].index("(")  # 文字列位置
                im4 = data_c_atc[:im2].index(")")  # 文字列位置
                d_at = float(data_c_atc[im3 + 1 : im4])  # 主筋断面積(X)
            else:
                d_hon = float(data_c_atc[:im2])  # 主筋本数(X)
        else:
            d_hon = float(data_c_atc[:im])  # 主筋本数(X)

        str_kei = data_c_atc[im + 1 :]  # 主筋径(X)
    else:
        if "(" in data_c_atc:
            # (主筋断面積) 径:(####)D## の処理
            im = data_c_atc.index("(")  # 文字列位置
            im2 = data_c_atc.index(")")  # 文字列位置
            d_at = float(data_c_atc[im + 1 : im2])  # 主筋断面積(X)
            str_kei = data_c_atc[im2 + 1 :]  # 主筋径(X)

    return (
        d_hon,
        d_at,
        str_kei,
    )


def get_colm(
    list_colm: list, str_zz: str, str_xx: str, str_yy: str, str_tb: str
) -> Tuple[
    float,
    float,
    str,
    float,
    float,
    str,
    float,
    float,
    str,
    str,
    str,
    float,
    float,
    float,
    float,
    str,
    float,
    float,
    float,
    str,
    float,
    str,
    str,
]:
    """柱部材断面情報のデータ取得

    柱部材断面情報のデータリストから計算に必要な柱データを取得します。

    Args:
        list_colm (list): 柱部材断面情報のデータリスト
        str_zz (str): 柱の位置  階名検索文字列
        str_xx (str): 柱の位置  X軸名検索文字列
        str_yy (str): 柱の位置  Y軸名検索文字列
        str_tb (str): 断面位置  検索文字列
    Returns:
        float: 柱せいDx
        float: 柱せいDy
        str: コンクリート材料
        float: 鉄筋本数(X)
        float: 主筋断面積(X)
        str: 鉄筋径(X)
        float: 鉄筋本数(Y)
        float: 主筋断面積(Y)
        str: 鉄筋径(Y)
        str: 鉄筋材料(X)
        str: 鉄筋材料(Y)
        float: dt(X)
        float: dt(Y)
        float: 帯筋本数(X)
        float: 帯筋断面積(X)
        str: 帯筋径(X)
        float: 帯筋ピッチ(X)
        float: 帯筋本数(Y)
        float: 帯筋断面積(Y)
        str: 帯筋径(Y)
        float: 帯筋ピッチ(Y)
        str: 帯筋材料(X)
        str: 帯筋材料(Y)
    Note:
        長方形断面のみ対応可(丸柱は扱わない)
        [柱の鉄筋位置]の入力方法を“かぶり”としても、1段目dt1として扱う
        柱主筋はX方向鉄筋径とY方向鉄筋径とも、同じ径を扱う
        直交配筋は考慮しない

    """

    # 柱部材断面情報のデータ取得
    d_dx: float = 0.0  # 柱せいDx
    d_dy: float = 0.0  # 柱せいDy
    str_zaifc: str = " "  # コンクリート材料
    d_hon_x: float = 0.0  # 鉄筋本数(X)
    d_at_x: float = 0.0  # 主筋断面積(X)
    str_kei_x: str = " "  # 鉄筋径(X)
    d_hon_y: float = 0.0  # 鉄筋本数(Y)
    d_at_y: float = 0.0  # 主筋断面積(Y)
    str_kei_y: str = " "  # 鉄筋径(Y)
    str_tekinzai_x: str = " "  # 鉄筋材料(X)
    str_tekinzai_y: str = " "  # 鉄筋材料(Y)
    d_dt_x: float = 0.0  # dt(X)
    d_dt_y: float = 0.0  # dt(Y)
    d_hoop_hon_x: float = 0.0  # 帯筋本数(X)
    d_hoop_at_x: float = 0.0  # 帯筋断面積(X)
    str_hoop_kei_x: str = " "  # 帯筋径(X)
    d_hoop_pich_x: float = 0.0  # 帯筋ピッチ(X)
    d_hoop_hon_y: float = 0.0  # 帯筋本数(Y)
    d_hoop_at_y: float = 0.0  # 帯筋断面積(Y)
    str_hoop_kei_y: str = " "  # 帯筋径(Y)
    d_hoop_pich_y: float = 0.0  # 帯筋ピッチ(Y)
    str_hoopzai_x: str = " "  # 帯筋材料(X)
    str_hoopzai_y: str = " "  # 帯筋材料(Y)

    i_nn = len(list_colm)
    for i in range(i_nn):
        if str_zz in list_colm[i] and str_xx in list_colm[i] and str_yy in list_colm[i]:
            # 以下、計算対象の柱データの取得

            # Dx×Dy
            data_c = list_colm[i]
            str_bd = re.findall(r"\d+", data_c[5])
            d_dx = float(str_bd[0])  # Dx
            d_dy = float(str_bd[1])  # Dy

            # 材料
            str_zaifc = data_c[6]  # コンクリート種別

            # 柱断面は長方形断面の場合
            # 柱頭
            i_ud: int = 0
            if str_tb.upper() == "B":  # 小文字は大文字へ変換する
                # 柱脚 (柱脚開始位置は、柱頭開始番号から10番目)
                i_ud = 10

            # 柱の主筋本数、主筋断面積 および 径
            # X方向
            data_c_atc = get_atc(data_c[7 + i_ud])

            # 取得した柱データを解析方向ごとの変数へ
            d_hon_x = data_c_atc[0]  # 鉄筋本数(X)
            d_at_x = data_c_atc[1]  # 主筋断面積(X)
            str_kei_x = data_c_atc[2]  # 鉄筋径(X)

            # 柱の主筋本数、主筋断面積 および 径
            # Y方向
            data_c_atc = get_atc(data_c[8 + i_ud])

            # 取得した柱データを解析方向ごとの変数へ
            d_hon_y = data_c_atc[0]  # 鉄筋本数(Y)
            d_at_y = data_c_atc[1]  # 主筋断面積(Y)
            str_kei_y = data_c_atc[2]  # 鉄筋径(Y)

            # 主筋材料
            str_tekinzai_x = data_c[9 + i_ud]  # 鉄筋材料(X)
            str_tekinzai_y = data_c[10 + i_ud]  # 鉄筋材料(Y)

            # 1段目dt
            d_dt_x = float(data_c[11 + i_ud])  # dt(X)
            d_dt_y = float(data_c[12 + i_ud])  # dt(Y)

            # 柱の帯筋本数、帯筋断面積、径 および ピッチ
            # X方向
            data_c_hoop = get_hoop(data_c[13 + i_ud])

            # 帯筋本数-径@ピッチ
            d_hoop_hon_x = data_c_hoop[0]  # 帯筋本数(X)
            d_hoop_at_x = data_c_hoop[1]  # 帯筋断面積(X)
            str_hoop_kei_x = data_c_hoop[2]  # 帯筋径(X)
            d_hoop_pich_x = data_c_hoop[3]  # 帯筋ピッチ(X)

            # 柱の帯筋本数、帯筋断面積、径 および ピッチ
            # Y方向
            data_c_hoop = get_hoop(data_c[14 + i_ud])

            # 帯筋本数-径@ピッチ
            d_hoop_hon_y = data_c_hoop[0]  # 帯筋本数(Y)
            d_hoop_at_y = data_c_hoop[1]  # 帯筋断面積(Y)
            str_hoop_kei_y = data_c_hoop[2]  # 帯筋径(Y)
            d_hoop_pich_y = data_c_hoop[3]  # 帯筋ピッチ(Y)

            # 帯筋材料
            str_hoopzai_x = data_c[15 + i_ud]  # 帯筋材料(X)
            str_hoopzai_y = data_c[16 + i_ud]  # 帯筋材料(X)

    return (
        d_dx,
        d_dy,
        str_zaifc,
        d_hon_x,
        d_at_x,
        str_kei_x,
        d_hon_y,
        d_at_y,
        str_kei_y,
        str_tekinzai_x,
        str_tekinzai_y,
        d_dt_x,
        d_dt_y,
        d_hoop_hon_x,
        d_hoop_at_x,
        str_hoop_kei_x,
        d_hoop_pich_x,
        d_hoop_hon_y,
        d_hoop_at_y,
        str_hoop_kei_y,
        d_hoop_pich_y,
        str_hoopzai_x,
        str_hoopzai_y,
    )


def get_fc(list_conc: list, str_zaifc: str) -> float:
    """コンクリート強度を取得

    コンクリート材料のデータリストから計算に必要なコンクリート強度を取得します。

    Args:
        list_conc (list): コンクリート材料のデータリスト
        str_zaifc (str): コンクリート材料検索文字列
    Returns:
        float: コンクリート強度

    """

    # コンクリート材料のデータ取得
    d_fc: float = 0.0
    i_nn = len(list_conc)
    for i in range(i_nn):
        if str_zaifc in list_conc[i]:
            d_fc = float(list_conc[i][2])  # コンクリートFc

    return d_fc


def get_at(list_tekinkei: list, str_kei_x: str, str_kei_y: str) -> Tuple[
    float,
    float,
]:
    """鉄筋径における鉄筋断面積を取得

    鉄筋径における鉄筋断面積のデータリストから計算に必要な鉄筋径における鉄筋断面積を取得します。

    Args:
        list_tekinkei (list): 鉄筋径における鉄筋断面積のデータリスト
        str_kei_x (str): X方向の鉄筋径検索文字列
        str_kei_y (str): Y方向の鉄筋径検索文字列
    Returns:
        float: 鉄筋径における鉄筋断面積(X)
        float: 鉄筋径における鉄筋断面積(Y)

    """

    # 鉄筋径における鉄筋断面積のデータ取得
    d_at1_x: float = 0.0
    d_at1_y: float = 0.0
    i_nn = len(list_tekinkei)
    for i in range(i_nn):
        if str_kei_x in list_tekinkei[i]:
            d_at1_x = float(list_tekinkei[i][4])  # 主筋径の断面積(X)

        if str_kei_y in list_tekinkei[i]:
            d_at1_y = float(list_tekinkei[i][4])  # 主筋径の断面積(Y)

    return (
        d_at1_x,
        d_at1_y,
    )


def get_fti(list_tekin: list, str_tekinzai_x: str, str_tekinzai_y: str) -> Tuple[
    float,
    float,
]:
    """主筋降伏点強度を取得

    鉄筋材料の降伏点強度のデータリストから計算に必要な主筋降伏点強度を取得します。

    Args:
        list_tekin (list): 鉄筋径における鉄筋断面積のデータリスト
        str_tekinzai_x (str): X方向の鉄筋材料検索文字列
        str_tekinzai_y (str): Y方向の鉄筋材料検索文字列
    Returns:
        float: 主筋降伏点強度(X)
        float: 主筋降伏点強度(Y)

    """

    # 鉄筋材料の降伏点強度のデータ取得
    d_fti_x: float = 0.0
    d_fti_y: float = 0.0
    i_nn = len(list_tekin)
    for i in range(i_nn):
        if str_tekinzai_x in list_tekin[i]:
            d_fti_x = float(list_tekin[i][1])  # 主筋降伏点強度(X)

        if str_tekinzai_y in list_tekin[i]:
            d_fti_y = float(list_tekin[i][1])  # 主筋降伏点強度(Y)

    return (
        d_fti_x,
        d_fti_y,
    )


def get_hoop(data_c_hoop: str) -> Tuple[
    float,
    float,
    str,
    float,
]:
    """柱の帯筋本数、帯筋断面積、径 および ピッチのデータ取得

    柱部材断面情報の帯筋データから柱の帯筋本数、帯筋断面積、径 および ピッチを取得します。

    Args:
        data_c_hoop (str): 柱部材断面情報の主筋データ
    Returns:
        float: 帯筋本数
        float: 帯筋断面積
        str: 帯筋径
        float: 帯筋ピッチ

    """

    # 柱の帯筋本数、帯筋断面積、径 および ピッチのデータ取得
    d_hoop_hon: float = 0.0  # 帯筋本数
    d_hoop_at: float = 0.0  # 帯筋断面積
    str_hoop_kei: str = " "  # 帯筋径
    d_hoop_pich: float = 0.0  # 帯筋ピッチ

    # 主筋本数-径
    if "-" in data_c_hoop:
        # 主筋本数-径:##-D## の処理
        im = data_c_hoop.index("-")  # 文字列位置

        if "@" in data_c_hoop:
            im2 = data_c_hoop.index("@")  # 文字列位置

        # 寄筋の指定
        d_hoop_hon = float(data_c_hoop[:im])  # 帯筋本数(X)
        str_hoop_kei = data_c_hoop[im + 1 : im2]  # 帯筋径(X)
        d_hoop_pich = float(data_c_hoop[im2 + 1 :])  # 帯筋ピッチ(X)

    else:
        if "(" in data_c_hoop:
            # (帯筋断面積) 径:(####)D## の処理
            im = data_c_hoop.index("(")  # 文字列位置
            im2 = data_c_hoop.index(")")  # 文字列位置

            if "@" in data_c_hoop:
                im3 = data_c_hoop.index("@")  # 文字列位置

            d_hoop_at = float(data_c_hoop[im + 1 : im2])  # 帯筋断面積(X)
            str_hoop_kei = data_c_hoop[im2 + 1 : im3]  # 帯筋径(X)
            d_hoop_pich = float(data_c_hoop[im3 + 1 :])  # 帯筋ピッチ(X)

    return (
        d_hoop_hon,
        d_hoop_at,
        str_hoop_kei,
        d_hoop_pich,
    )


def get_mu(
    data_c_t: Tuple,
    data_c_b: Tuple,
    list_conc: list,
    list_tekin: list,
    list_tekinkei: list,
    str_xy: str,
) -> Tuple[
    float,
    float,
    float,
    float,
    float,
    float,
    float,
]:
    """曲げ終局強度の結果取得

    曲げ終局強度の結果を取得します。

    Args:
        data_c_t (Tuple): 柱頭側断面情報のデータ
        data_c_b (Tuple): 柱脚側断面情報のデータ
        list_conc (list): コンクリート材料のデータ
        list_tekin (list): 鉄筋材料の降伏点強度のデータ
        list_tekinkei (list): 鉄筋径における鉄筋断面積のデータ
        str_xy (str): 解析方向の入力 X方向 または Y方向
    Returns:
        float: 柱頭側ピース断面分割のせい
        float: 柱頭側中立軸
        float: 柱頭側曲げ終局強度
        float: 柱脚側ピース断面分割のせい
        float: 柱脚側中立軸
        float: 柱脚側曲げ終局強度
        float: 柱せい
    Note:

    """

    for i_lp in range(2):

        # 取得した柱データを解析方向ごとの変数へ
        if i_lp == 0:
            d_dx = data_c_t[0]  # 柱せいDx
            d_dy = data_c_t[1]  # 柱せいDy
            str_zaifc = data_c_t[2]  # コンクリート材料
            d_hon_x = data_c_t[3]  # 鉄筋本数(X)
            d_at_x = data_c_t[4]  # 主筋断面積(X)
            str_kei_x = data_c_t[5]  # 鉄筋径(X)
            d_hon_y = data_c_t[6]  # 鉄筋本数(Y)
            d_at_y = data_c_t[7]  # 主筋断面積(Y)
            str_kei_y = data_c_t[8]  # 鉄筋径(Y)
            str_tekinzai_x = data_c_t[9]  # 鉄筋材料(X)
            str_tekinzai_y = data_c_t[10]  # 鉄筋材料(Y)
            d_dt_x = data_c_t[11]  # dt(X)
            d_dt_y = data_c_t[12]  # dt(Y)
            str_hoop_kei_x = data_c_t[15]  # 帯筋径(X)
            str_hoop_kei_y = data_c_t[19]  # 帯筋径(Y)
            str_hoopzai_x = data_c_t[21]  # 帯筋材料(X)
            str_hoopzai_y = data_c_t[22]  # 帯筋材料(Y)
        else:
            d_dx = data_c_b[0]  # 柱せいDx
            d_dy = data_c_b[1]  # 柱せいDy
            str_zaifc = data_c_b[2]  # コンクリート材料
            d_hon_x = data_c_b[3]  # 鉄筋本数(X)
            d_at_x = data_c_b[4]  # 主筋断面積(X)
            str_kei_x = data_c_b[5]  # 鉄筋径(X)
            d_hon_y = data_c_b[6]  # 鉄筋本数(Y)
            d_at_y = data_c_b[7]  # 主筋断面積(Y)
            str_kei_y = data_c_b[8]  # 鉄筋径(Y)
            str_tekinzai_x = data_c_b[9]  # 鉄筋材料(X)
            str_tekinzai_y = data_c_b[10]  # 鉄筋材料(Y)
            d_dt_x = data_c_b[11]  # dt(X)
            d_dt_y = data_c_b[12]  # dt(Y)
            str_hoop_kei_x = data_c_b[15]  # 帯筋径(X)
            str_hoop_kei_y = data_c_b[19]  # 帯筋径(Y)
            str_hoopzai_x = data_c_b[21]  # 帯筋材料(X)
            str_hoopzai_y = data_c_b[22]  # 帯筋材料(Y)

        # コンクリート材料のデータを取得する
        d_fc = get_fc(list_conc, str_zaifc)

        # 鉄筋径における鉄筋断面積のデータを取得する
        ad_at1 = get_at(list_tekinkei, str_kei_x, str_kei_y)

        # 取得した鉄筋径における鉄筋断面積を解析方向ごとの変数へ
        d_at1_x = ad_at1[0]  # 鉄筋径における鉄筋断面積(X)
        d_at1_y = ad_at1[1]  # 鉄筋径における鉄筋断面積(Y)

        # 鉄筋本数入力による鉄筋断面積を算定する
        if d_hon_x > 0.0:
            d_at_x = d_hon_x * d_at1_x  # 主筋断面積(X)

        if d_hon_y > 0.0:
            d_at_y = d_hon_y * d_at1_y  # 主筋断面積(Y)

        # 鉄筋材料の降伏点強度のデータを取得する
        ad_fti = get_fti(list_tekin, str_tekinzai_x, str_tekinzai_y)

        # 取得した鉄筋材料の降伏点強度を解析方向ごとの変数へ
        d_fti_x = ad_fti[0]  # 主筋降伏点強度(X)
        d_fti_y = ad_fti[1]  # 主筋降伏点強度(Y)

        # 鉄筋径における鉄筋断面積のデータを取得する
        ad_at1 = get_at(list_tekinkei, str_hoop_kei_x, str_hoop_kei_y)

        # 鉄筋材料の降伏点強度のデータを取得する
        ad_fti = get_fti(list_tekin, str_hoopzai_x, str_hoopzai_y)

        # 算定式のパラメータに柱データを格納する
        # X方向
        d_bb = d_dy  # 柱幅
        d_dd = d_dx  # 柱せい
        d_at = d_at_x  # 主筋断面積
        d_sy = d_fti_x  # 主筋降伏点強度
        d_dt = d_dt_x  # dt
        if str_xy.upper() == "Y":  # 小文字は大文字へ変換する
            # Y方向
            d_bb = d_dx  # 柱幅
            d_dd = d_dy  # 柱せい
            d_at = d_at_y  # 主筋断面積
            d_sy = d_fti_y  # 主筋降伏点強度
            d_dt = d_dt_y  # dt

        d_dl2: float = 0.0  # ピース断面分割のせい
        d_z: float = 0.0  # 中立軸
        d_mu: float = 0.0  # 曲げ終局強度

        # ピース断面分割による曲げ終局強度の計算
        # (柱断面、コンクリート強度、dtのデータは必須データ)
        if d_bb * d_dd * d_dt * d_fc != 0.0:
            # 曲げ終局強度の計算開始
            ad_mu = calc_mu(d_bb, d_dd, d_at, d_sy, d_dt, d_fc, d_sn)

            # 取得した計算結果を出力用変数へ
            d_dl2 = ad_mu[0]  # ピース断面分割のせい
            d_z = ad_mu[1]  # 中立軸
            d_mu = ad_mu[2]  # 曲げ終局強度

        else:
            # 柱断面、コンクリート強度、dtのデータで0(ゼロ)が存在するため、計算できない
            # (計算に必要なパラメータがない)
            d_dl2 = 0.0
            d_z = -1.0  # 計算に必要なパラメータがないフラグとして、負値(-1)を設定する

        if i_lp == 0:
            d_tdl2 = d_dl2
            d_tz = d_z
            d_tmu = d_mu
        else:
            d_bdl2 = d_dl2
            d_bz = d_z
            d_bmu = d_mu

    return (
        d_tdl2,
        d_tz,
        d_tmu,
        d_bdl2,
        d_bz,
        d_bmu,
        d_dd,
    )


def get_qsu(
    data_c_t: Tuple,
    data_c_b: Tuple,
    list_conc: list,
    list_tekin: list,
    list_tekinkei: list,
    d_ho: float,
    d_sn: float,
    d_tmu: float,
    d_bmu: float,
    str_xy: str,
) -> Tuple[
    float,
    float,
]:
    """せん断終局強度の結果取得

    せん断終局強度の結果を取得します。

    Args:
        data_c_t (Tuple): 柱頭側断面情報のデータ
        data_c_b (Tuple): 柱脚側断面情報のデータ
        list_conc (list): コンクリート材料のデータ
        list_tekin (list): 鉄筋材料の降伏点強度のデータ
        list_tekinkei (list): 鉄筋径における鉄筋断面積のデータ
        d_ho (float): 内法高さ
        d_sn (float): 軸力
        d_tmu (float): 柱頭側曲げ終局強度
        d_bmu (float): 柱脚側曲げ終局強度
        str_xy (str): 解析方向の入力 X方向 または Y方向
    Returns:
        float: 柱頭側せん断終局強度
        float: 柱脚側せん断終局強度
    Note:

    """

    for i_lp in range(2):

        # 取得した柱データを解析方向ごとの変数へ
        if i_lp == 0:
            str_tb = "T"  # 柱頭
            d_dx = data_c_t[0]  # 柱せいDx
            d_dy = data_c_t[1]  # 柱せいDy
            str_zaifc = data_c_t[2]  # コンクリート材料
            d_hon_x = data_c_t[3]  # 鉄筋本数(X)
            d_at_x = data_c_t[4]  # 主筋断面積(X)
            str_kei_x = data_c_t[5]  # 鉄筋径(X)
            d_hon_y = data_c_t[6]  # 鉄筋本数(Y)
            d_at_y = data_c_t[7]  # 主筋断面積(Y)
            str_kei_y = data_c_t[8]  # 鉄筋径(Y)
            str_tekinzai_x = data_c_t[9]  # 鉄筋材料(X)
            str_tekinzai_y = data_c_t[10]  # 鉄筋材料(Y)
            d_dt_x = data_c_t[11]  # dt(X)
            d_dt_y = data_c_t[12]  # dt(Y)
            d_hoop_hon_x = data_c_t[13]  # 帯筋本数(X)
            d_hoop_at_x = data_c_t[14]  # 帯筋断面積(X)
            str_hoop_kei_x = data_c_t[15]  # 帯筋径(X)
            d_hoop_pich_x = data_c_t[16]  # 帯筋ピッチ(X)
            d_hoop_hon_y = data_c_t[17]  # 帯筋本数(Y)
            d_hoop_at_y = data_c_t[18]  # 帯筋断面積(Y)
            str_hoop_kei_y = data_c_t[19]  # 帯筋径(Y)
            d_hoop_pich_y = data_c_t[20]  # 帯筋ピッチ(Y)
            str_hoopzai_x = data_c_t[21]  # 帯筋材料(X)
            str_hoopzai_y = data_c_t[22]  # 帯筋材料(Y)
        else:
            str_tb = "B"  # 柱脚
            d_dx = data_c_b[0]  # 柱せいDx
            d_dy = data_c_b[1]  # 柱せいDy
            str_zaifc = data_c_b[2]  # コンクリート材料
            d_hon_x = data_c_b[3]  # 鉄筋本数(X)
            d_at_x = data_c_b[4]  # 主筋断面積(X)
            str_kei_x = data_c_b[5]  # 鉄筋径(X)
            d_hon_y = data_c_b[6]  # 鉄筋本数(Y)
            d_at_y = data_c_b[7]  # 主筋断面積(Y)
            str_kei_y = data_c_b[8]  # 鉄筋径(Y)
            str_tekinzai_x = data_c_b[9]  # 鉄筋材料(X)
            str_tekinzai_y = data_c_b[10]  # 鉄筋材料(Y)
            d_dt_x = data_c_b[11]  # dt(X)
            d_dt_y = data_c_b[12]  # dt(Y)
            d_hoop_hon_x = data_c_b[13]  # 帯筋本数(X)
            d_hoop_at_x = data_c_b[14]  # 帯筋断面積(X)
            str_hoop_kei_x = data_c_b[15]  # 帯筋径(X)
            d_hoop_pich_x = data_c_b[16]  # 帯筋ピッチ(X)
            d_hoop_hon_y = data_c_b[17]  # 帯筋本数(Y)
            d_hoop_at_y = data_c_b[18]  # 帯筋断面積(Y)
            str_hoop_kei_y = data_c_b[19]  # 帯筋径(Y)
            d_hoop_pich_y = data_c_b[20]  # 帯筋ピッチ(Y)
            str_hoopzai_x = data_c_b[21]  # 帯筋材料(X)
            str_hoopzai_y = data_c_b[22]  # 帯筋材料(Y)

        # コンクリート材料のデータを取得する
        d_fc = get_fc(list_conc, str_zaifc)

        # 鉄筋径における鉄筋断面積のデータを取得する
        ad_at1 = get_at(list_tekinkei, str_kei_x, str_kei_y)

        # 取得した鉄筋径における鉄筋断面積を解析方向ごとの変数へ
        d_at1_x = ad_at1[0]  # 鉄筋径における鉄筋断面積(X)
        d_at1_y = ad_at1[1]  # 鉄筋径における鉄筋断面積(Y)

        # 鉄筋本数入力による鉄筋断面積を算定する
        if d_hon_x > 0.0:
            d_at_x = d_hon_x * d_at1_x  # 主筋断面積(X)

        if d_hon_y > 0.0:
            d_at_y = d_hon_y * d_at1_y  # 主筋断面積(Y)

        # 鉄筋材料の降伏点強度のデータを取得する
        ad_fti = get_fti(list_tekin, str_tekinzai_x, str_tekinzai_y)

        # 鉄筋径における鉄筋断面積のデータを取得する
        ad_at1 = get_at(list_tekinkei, str_hoop_kei_x, str_hoop_kei_y)

        # 取得した鉄筋径における鉄筋断面積を解析方向ごとの変数へ
        d_aw1_x = ad_at1[0]  # 鉄筋径における鉄筋断面積(X)
        d_aw1_y = ad_at1[1]  # 鉄筋径における鉄筋断面積(Y)

        # 鉄筋本数入力による鉄筋断面積を算定する
        if d_hoop_hon_x > 0.0:
            d_hoop_at_x = d_hoop_hon_x * d_aw1_x  # 帯筋断面積(X)

        if d_hoop_hon_y > 0.0:
            d_hoop_at_y = d_hoop_hon_y * d_aw1_y  # 帯筋断面積(Y)

        # 鉄筋材料の降伏点強度のデータを取得する
        ad_fti = get_fti(list_tekin, str_hoopzai_x, str_hoopzai_y)

        # 取得した鉄筋材料の降伏点強度を解析方向ごとの変数へ
        d_swy_x = ad_fti[0]  # 帯筋降伏点強度(X)
        d_swy_y = ad_fti[1]  # 帯筋降伏点強度(Y)

        # 算定式のパラメータに柱データを格納する
        # X方向
        d_bb = d_dy  # 柱幅
        d_dd = d_dx  # 柱せい
        d_at = d_at_x  # 主筋断面積
        d_dt = d_dt_x  # dt
        d_aw = d_hoop_at_x  # 帯筋断面積
        d_swy = d_swy_x  # 帯筋降伏点強度
        d_pich = d_hoop_pich_x  # 帯筋ピッチ
        if str_xy.upper() == "Y":  # 小文字は大文字へ変換する
            # Y方向
            d_bb = d_dx  # 柱幅
            d_dd = d_dy  # 柱せい
            d_at = d_at_y  # 主筋断面積
            d_dt = d_dt_y  # dt
            d_aw = d_hoop_at_y  # 帯筋断面積
            d_swy = d_swy_y  # 帯筋降伏点強度
            d_pich = d_hoop_pich_y  # 帯筋ピッチ

        d_qsu: float = 0.0  # せん断終局強度

        # せん断終局強度の計算開始
        d_qsu = calc_qsu(
            d_bb,
            d_dd,
            d_at,
            d_aw,
            d_swy,
            d_pich,
            d_dt,
            d_fc,
            d_ho,
            d_sn,
            d_tmu,
            d_bmu,
            str_tb,
        )

        if i_lp == 0:
            d_tqsu = d_qsu
        else:
            d_bqsu = d_qsu

    return (
        d_tqsu,
        d_bqsu,
    )


def get_cff(
    data_c_t: Tuple,
    data_c_b: Tuple,
    list_conc: list,
    list_tekinkei: list,
    d_ho: float,
    d_hh: float,
    d_sn: float,
    d_tmu: float,
    d_bmu: float,
    d_tqsu: float,
    d_bqsu: float,
    str_xy: str,
) -> float:
    """# 柱のF指標の結果取得

    # 柱のF指標の結果を取得します。

    Args:
        data_c_t (Tuple): 柱頭側断面情報のデータ
        data_c_b (Tuple): 柱脚側断面情報のデータ
        list_conc (list): コンクリート材料のデータ
        list_tekin (list): 鉄筋材料の降伏点強度のデータ
        list_tekinkei (list): 鉄筋径における鉄筋断面積のデータ
        d_ho (float): 内法高さ
        d_hh (float): 標準内法高さ
        d_sn (float): 軸力
        d_tmu (float): 柱頭側曲げ終局強度
        d_bmu (float): 柱脚側曲げ終局強度
        d_tqsu (float): 柱頭側せん断終局強度
        d_bqsu (float): 柱脚側せん断終局強度
        str_xy (str): 解析方向の入力 X方向 または Y方向
    Returns:
        float: 柱のF指標
    Note:

    """

    for i_lp in range(2):

        # 取得した柱データを解析方向ごとの変数へ
        if i_lp == 0:
            d_dx = data_c_t[0]  # 柱せいDx
            d_dy = data_c_t[1]  # 柱せいDy
            str_zaifc = data_c_t[2]  # コンクリート材料
            d_hon_x = data_c_t[3]  # 鉄筋本数(X)
            d_at_x = data_c_t[4]  # 主筋断面積(X)
            str_kei_x = data_c_t[5]  # 鉄筋径(X)
            d_hon_y = data_c_t[6]  # 鉄筋本数(Y)
            d_at_y = data_c_t[7]  # 主筋断面積(Y)
            str_kei_y = data_c_t[8]  # 鉄筋径(Y)
            str_hoop_kei_x = data_c_t[15]  # 帯筋径(X)
            d_hoop_pich_x = data_c_t[16]  # 帯筋ピッチ(X)
            str_hoop_kei_y = data_c_t[19]  # 帯筋径(Y)
            d_hoop_pich_y = data_c_t[20]  # 帯筋ピッチ(Y)
        else:
            d_dx = data_c_b[0]  # 柱せいDx
            d_dy = data_c_b[1]  # 柱せいDy
            str_zaifc = data_c_b[2]  # コンクリート材料
            d_hon_x = data_c_b[3]  # 鉄筋本数(X)
            d_at_x = data_c_b[4]  # 主筋断面積(X)
            str_kei_x = data_c_b[5]  # 鉄筋径(X)
            d_hon_y = data_c_b[6]  # 鉄筋本数(Y)
            d_at_y = data_c_b[7]  # 主筋断面積(Y)
            str_kei_y = data_c_b[8]  # 鉄筋径(Y)
            str_hoop_kei_x = data_c_b[15]  # 帯筋径(X)
            d_hoop_pich_x = data_c_b[16]  # 帯筋ピッチ(X)
            str_hoop_kei_y = data_c_b[19]  # 帯筋径(Y)
            d_hoop_pich_y = data_c_b[20]  # 帯筋ピッチ(Y)

        # コンクリート材料のデータを取得する
        d_fc = get_fc(list_conc, str_zaifc)

        # 鉄筋径における鉄筋断面積のデータを取得する
        ad_at1 = get_at(list_tekinkei, str_kei_x, str_kei_y)

        # 取得した鉄筋径における鉄筋断面積を解析方向ごとの変数へ
        d_at1_x = ad_at1[0]  # 鉄筋径における鉄筋断面積(X)
        d_at1_y = ad_at1[1]  # 鉄筋径における鉄筋断面積(Y)

        # 鉄筋本数入力による鉄筋断面積を算定する
        if d_hon_x > 0.0:
            d_at_x = d_hon_x * d_at1_x  # 主筋断面積(X)

        if d_hon_y > 0.0:
            d_at_y = d_hon_y * d_at1_y  # 主筋断面積(Y)

        # 鉄筋径における鉄筋断面積のデータを取得する
        ad_at1 = get_at(list_tekinkei, str_hoop_kei_x, str_hoop_kei_y)

        # 算定式のパラメータに柱データを格納する
        # X方向
        d_bb = d_dy  # 柱幅
        d_dd = d_dx  # 柱せい
        d_at = d_at_x  # 主筋断面積
        d_pich = d_hoop_pich_x  # 帯筋ピッチ
        d_kei = float(str_kei_x[1:])  # 主筋径
        if str_xy.upper() == "Y":  # 小文字は大文字へ変換する
            # Y方向
            d_bb = d_dx  # 柱幅
            d_dd = d_dy  # 柱せい
            d_at = d_at_y  # 主筋断面積
            d_pich = d_hoop_pich_y  # 帯筋ピッチ
            d_kei = float(str_kei_y[1:])  # 主筋径

        if i_lp == 0:
            d_tat = d_at
            d_tkei = d_kei
        else:
            d_bat = d_at
            d_bkei = d_kei

    d_at = max(d_tat, d_bat)
    d_kei = min(d_tkei, d_bkei)
    d_cff: float = 0.0  # せん断終局強度

    # 柱のF指標の計算開始
    d_cff = get_result_cff(
        d_bb,
        d_dd,
        d_at,
        d_kei,
        d_pich,
        d_fc,
        d_ho,
        d_hh,
        d_sn,
        d_tmu,
        d_bmu,
        d_tqsu,
        d_bqsu,
    )

    return d_cff


def calc_mu(
    d_bb: float,
    d_dd: float,
    d_at: float,
    d_sy: float,
    d_dt: float,
    d_fc: float,
    d_sn: float,
) -> Tuple[
    float,
    float,
    float,
]:
    """ピース断面分割による曲げ終局強度の計算

    ピース断面分割による曲げ終局強度を計算します。

    Args:
        d_bb (float): 柱幅
        d_dd (float): 柱せい
        d_at (float): 主筋断面積
        d_sy (float): 主筋降伏点強度
        d_dt (float): dt
        d_fc (float): コンクリート強度
        d_sn (float): 軸力
    Returns:
        float: ピース断面(2)の柱せい
        float: 中立軸
        float: 曲げ終局強度
    Note:
        柱断面、コンクリート強度、dtのデータは必須

    """

    # ピース断面分割による曲げ終局強度の計算
    # 柱幅
    d_bl1 = d_bb
    d_bl2 = d_bl1
    d_bl3 = d_bl1

    # 柱せい
    d_dl1 = d_dt * 2.0
    d_dl3 = d_dl1
    d_dl2 = d_dd - d_dl1 - d_dl3

    # ピース断面までの距離
    d_dk1 = d_dl1
    d_dk2 = d_dl1 + d_dl2
    d_dk3 = d_dl1 + d_dl2 + d_dl3

    # 重心位置
    d_ll1 = 0.5 * d_dl1
    d_ll2 = 0.5 * d_dl2 + d_dl1
    d_ll3 = 0.5 * d_dl3 + d_dl1 + d_dl2

    # 単位長さあたりの主筋断面積
    try:
        d_sl1 = d_at / d_dl1
    except ZeroDivisionError:
        d_sl1 = 0.0

    d_sl3 = d_sl1
    d_sl2 = 0.0

    # 単位長さあたりの主筋断面積×主筋降伏点強度
    d_slsy1 = d_sl1 * d_sy
    d_slsy3 = d_sl3 * d_sy
    d_slsy2 = d_sl2 * d_sy

    # 主筋断面積×主筋降伏点強度
    d_sldlsy1 = d_slsy1 * d_dl1
    d_sldlsy2 = d_slsy2 * d_dl2
    d_sldlsy3 = d_slsy3 * d_dl3

    # 軸力(入力値)
    d_sn = d_sn * 1000.0

    # 軸力の作用位置
    d_ln = 0.5 * d_dd

    # 主筋断面積×主筋降伏点強度の集計
    d_sldlsy = d_sldlsy1 + d_sldlsy2 + d_sldlsy3

    # ピース断面(3)に中立軸があると仮定したとき
    # 中立軸の計算
    d_z3 = (
        (-1.0)
        * (d_sldlsy + d_sn - (2.0 * d_slsy3 + d_bl3 * d_fc) * d_dk3)
        / (2.0 * d_slsy3 + d_bl3 * d_fc)
    )

    # 曲げ終局強度の計算
    d_mu3 = (
        (2.0 * d_slsy3 + d_bl3 * d_fc) * (d_dk3 - d_z3) * (d_dk3 + d_z3) * 0.5
        - d_sldlsy1 * d_ll1
        - d_sldlsy2 * d_ll2
        - d_sldlsy3 * d_ll3
        - d_sn * d_ln
    )
    d_mu3 = d_mu3 / 1000000.0

    # ピース断面(2)に中立軸があると仮定したとき
    # 中立軸の計算
    d_z2 = (
        (-1.0)
        * (
            (d_sldlsy + d_sn - (2.0 * d_slsy2 + d_bl2 * d_fc) * d_dk2)
            - (2.0 * d_sldlsy3 + d_bl3 * d_dl3 * d_fc)
        )
        / (2.0 * d_slsy2 + d_bl2 * d_fc)
    )

    # 曲げ終局強度の計算
    d_mu2 = (
        (2.0 * d_slsy2 + d_bl2 * d_fc) * (d_dk2 - d_z2) * (d_dk2 + d_z2) * 0.5
        + (2.0 * d_sldlsy3 + d_bl3 * d_dl3 * d_fc) * d_ll3
        - d_sldlsy1 * d_ll1
        - d_sldlsy2 * d_ll2
        - d_sldlsy3 * d_ll3
        - d_sn * d_ln
    )
    d_mu2 = d_mu2 / 1000000.0

    # ピース断面(1)に中立軸があると仮定したとき
    # 中立軸の計算
    d_z1 = (
        (-1.0)
        * (
            (d_sldlsy + d_sn - (2.0 * d_slsy1 + d_bl1 * d_fc) * d_dk1)
            - (2.0 * d_sldlsy2 + d_bl2 * d_dl2 * d_fc)
            - (2.0 * d_sldlsy3 + d_bl3 * d_dl3 * d_fc)
        )
        / (2.0 * d_slsy1 + d_bl1 * d_fc)
    )

    # 曲げ終局強度の計算
    d_mu1 = (
        (2.0 * d_slsy1 + d_bl1 * d_fc) * (d_dk1 - d_z1) * (d_dk1 + d_z1) * 0.5
        + (2.0 * d_sldlsy2 + d_bl2 * d_dl2 * d_fc) * d_ll2
        + (2.0 * d_sldlsy3 + d_bl3 * d_dl3 * d_fc) * d_ll3
        - d_sldlsy1 * d_ll1
        - d_sldlsy2 * d_ll2
        - d_sldlsy3 * d_ll3
        - d_sn * d_ln
    )
    d_mu1 = d_mu1 / 1000000.0

    # 中立軸がある断面の曲げ終局強度を採用する
    d_z: float = -1.0  # 計算ができない場合のフラグとして、負値(-1)を設定する
    d_mu: float = 0.0
    if d_dk2 < d_z3 <= d_dk3:
        d_z = d_z3
        d_mu = math.floor(d_mu3 * 10.0) / 10.0  # 小数点第1位までを表示させる

    if d_dk1 < d_z2 <= d_dk2:
        d_z = d_z2
        d_mu = math.floor(d_mu2 * 10.0) / 10.0  # 小数点第1位までを表示させる

    if 0.0 <= d_z1 <= d_dk1:
        d_z = d_z1
        d_mu = math.floor(d_mu1 * 10.0) / 10.0  # 小数点第1位までを表示させる

    return (
        d_dl2,
        d_z,
        d_mu,
    )


def calc_qsu(
    d_bb: float,
    d_dd: float,
    d_at: float,
    d_aw: float,
    d_swy: float,
    d_pich: float,
    d_dt: float,
    d_fc: float,
    d_ho: float,
    d_sn: float,
    d_tmu: float,
    d_bmu: float,
    str_tb: str,
) -> float:
    """せん断終局強度の計算

    せん断終局強度を計算します。

    Args:
        d_bb (float): 柱幅
        d_dd (float): 柱せい
        d_at (float): 主筋断面積
        d_aw (float): 帯筋断面積
        d_swy (float): 帯筋降伏点強度
        d_pich (float): 帯筋ピッチ
        d_dt (float): dt
        d_fc (float): コンクリート強度
        d_ho (float): 柱内法高さ
        d_sn (float): 軸力
        d_tmu (float): 柱頭の曲げ終局強度
        d_bmu (float): 柱脚の曲げ終局強度
        str_tb (str): 断面位置  検索文字列
    Returns:
        float: せん断終局強度
    Note:

    """

    # せん断終局強度の計算
    # 柱幅
    d_bl1 = d_bb

    # 柱の有効せい
    d_dl1 = d_dd - d_dt

    # 柱の応力中心間距離
    d_jj1 = 0.8 * d_dd

    # 柱の引張鉄筋比
    d_pt1 = d_at * 100.0 / (d_bl1 * d_dd)

    if d_pt1 < 0.1:
        d_pt1 = 0.1

    # 柱のせん断補強筋比
    d_pw1 = d_aw / (d_bl1 * d_pich)

    if d_pw1 > 0.012:
        d_pw1 = 0.012

    # 柱のせん断補強筋比×帯筋降伏点強度
    d_pwswy1 = d_pw1 * d_swy

    # 柱軸方向応力度
    d_sigma = d_sn * 1000.0 / (d_bl1 * d_dd)

    if d_sigma > 8.0:
        d_sigma = 8.0

    # 柱のせん断スパンM/Q
    if str_tb.upper() == "T":  # 小文字は大文字へ変換する
        # 柱頭
        d_mq = d_ho * d_tmu / (d_tmu + d_bmu)
    else:
        # 柱脚
        d_mq = d_ho * d_bmu / (d_tmu + d_bmu)

    # 柱のM/Qd
    d_mqd = d_mq / d_dl1

    if d_mqd < 1.0:
        d_mqd = 1.0

    if d_mqd > 3.0:
        d_mqd = 3.0

    # せん断終局強度の計算
    d_qsu1 = (0.053 * (d_pt1) ** 0.23 * (18 + d_fc)) / (d_mqd + 0.12)

    d_qsu2 = 0.85 * pow(d_pwswy1, 0.5)

    d_qsu3 = 0.1 * d_sigma

    d_qsu = (d_qsu1 + d_qsu2 + d_qsu3) * d_bl1 * d_jj1
    d_qsu = d_qsu / 1000.0
    d_qsu = math.floor(d_qsu * 10.0) / 10.0  # 小数点第1位までを表示させる

    return d_qsu


def calc_crmax(
    d_bb: float,
    d_dd: float,
    d_at: float,
    d_kei: float,
    d_pich: float,
    d_fc: float,
    d_ho: float,
    d_sn: float,
    d_qmu: float,
    d_qsu: float,
) -> float:
    """柱の曲げ終局変形角の上限cRmaxの計算

    柱の曲げ終局変形角の上限cRmaxを計算します。

    Args:
        d_bb (float): 柱幅
        d_dd (float): 柱せい
        d_at (float): 主筋断面積
        d_kei (float): 主筋径
        d_pich (float): 帯筋ピッチ
        d_fc (float): コンクリート強度
        d_ho (float): 柱内法高さ
        d_sn (float): 軸力
        d_qmu (float): 曲げ終局時せん断力
        d_qsu (float): せん断終局強度
    Returns:
        float: 柱の曲げ終局変形角の上限cRmax
    Note:

    """

    # 柱の曲げ終局変形角の上限cRmax
    # 柱幅
    d_bl1 = d_bb

    # 柱の応力中心間距離
    d_jj1 = 0.8 * d_dd

    # 変形角の定数
    d_r250 = 1 / 250.0
    d_r50 = 1 / 50.0
    d_r30 = 1 / 30.0

    # 軸力による上限cRmaxn
    if d_pich <= 100.0:
        d_etal = 0.25
        d_etah = 0.50
    else:
        d_etal = 0.20
        d_etah = 0.40

    d_nsbdfc = d_sn * 1000.0 / (d_bl1 * d_dd * d_fc)

    d_ndash = (d_nsbdfc - d_etal) / (d_etah - d_etal)
    d_crmaxn = d_r30 * (d_r250 / d_r30) ** d_ndash

    if d_crmaxn > d_r30:
        d_crmaxn = d_r30

    if d_nsbdfc > d_etah:
        d_crmaxn = d_r250

    # せん断応力度による上限cRmaxs
    # 曲げ終局時せん断力
    d_tu1 = d_qmu * 1000.0 / (d_bl1 * d_jj1 * d_fc)
    # せん断終局強度
    d_tu2 = d_qsu * 1000.0 / (d_bl1 * d_jj1 * d_fc)

    if d_tu1 > d_tu2:
        d_tu = d_tu2
    else:
        d_tu = d_tu1

    if d_tu > 0.20:
        d_crmaxs = d_r250
    else:
        d_crmaxs = d_r30

    # 引張鉄筋比による曲げ終局時変形角の上限cRmaxt
    d_pt1 = d_at * 100.0 / (d_bl1 * d_dd)

    if d_pt1 > 1.0:
        d_crmaxt = d_r250
    else:
        d_crmaxt = d_r30

    # 帯筋間隔による曲げ終局時変形角の上限cRmaxb
    d_pichkei = d_pich / d_kei

    if d_pichkei > 8.0:
        d_crmaxb = d_r50
    else:
        d_crmaxb = d_r30

    # 内法高さによる曲げ終局時変形角の上限cRmaxh
    d_hodd = d_ho / d_dd

    if d_hodd <= 2.0:
        d_crmaxh = d_r250
    else:
        d_crmaxh = d_r30

    # 上限cRmaxh
    list = []
    list.append(d_crmaxn)
    list.append(d_crmaxs)
    list.append(d_crmaxt)
    list.append(d_crmaxb)
    list.append(d_crmaxh)
    d_crmax = min(list)

    return d_crmax


def calc_crmy(d_hodd: float) -> float:
    """曲げ降伏時部材角cRmyの計算

    曲げ降伏時部材角cRmyを計算します。

    Args:
        d_hodd (float): ho/D
    Returns:
        float: 曲げ降伏時部材角cRmy
    Note:

    """

    # 変形角の定数
    d_r250 = 1 / 250.0
    d_r150 = 1 / 150.0

    # 曲げ降伏時部材角cRmy

    if d_hodd <= 2.0:
        d_crmy = d_r250
    elif d_hodd >= 3.0:
        d_crmy = d_r150
    else:
        d_crmy = d_r250 + (d_r150 - d_r250) * (d_hodd - 2.0)

    return d_crmy


def calc_crmp(
    d_qmu: float,
    d_qsu: float,
    d_qq: float,
    d_crmy: float,
) -> float:
    """曲げ塑性変形角cRmpの計算

    曲げ塑性変形角cRmpを計算します。

    Args:
        d_qmu (float): 曲げ終局強度時せん断力Qmu
        d_qsu (float): せん断終局強度Qsu
        d_qq (float): cRmp算定におけるq
        d_rmu (float): 曲げ終局時層間変形角Rmu
    Returns:
        float: 曲げ塑性変形角cRmp
    Note:

    """

    try:
        d_crmp = 10.0 * ((d_qsu / d_qmu) - d_qq) * d_crmy
    except ZeroDivisionError:
        d_crmp = 0.0

    if d_crmp < 0.0:
        d_crmp = 0.0

    return d_crmp


def calc_rsu(d_qmu: float, d_qsu: float, d_rmy: float) -> float:
    """せん断柱の終局時層間変形角Rsuの計算

    せん断柱の終局時層間変形角Rsuを計算します。

    Args:
        d_qmu (float): 曲げ終局時せん断力
        d_qsu (float): せん断終局強度
        d_rmy (float): 曲げ降伏時層間変形角Rmy
    Returns:
        float: せん断柱の終局時層間変形角Rsu
    Note:


    """

    # 変形角の定数
    d_r250 = 1 / 250.0

    if d_qmu > d_qsu:
        # せん断柱の強度寄与係数cα
        d_calfa = 0.3 + 0.7 * (d_r250 / d_rmy)

        # せん断柱の終局時層間変形角Rsu
        if d_calfa * d_qmu < d_qsu:
            d_rsu = (d_qsu / d_qmu - 0.3) / 0.7 * d_rmy

            if d_rsu < d_r250:
                d_rsu = d_r250
        else:
            d_rsu = d_r250

    return d_rsu


def calc_cff(
    d_ho: float,
    d_dd: float,
    d_qmu: float,
    d_qsu: float,
    d_rmu: float,
    d_rmy: float,
) -> float:
    """柱のF指標の計算

    柱のF指標を計算します。

    Args:
        d_ho (float): 柱内法高さ
        d_dd (float): 柱せい
        d_qmu (float): 曲げ終局強度時せん断力Qmu
        d_qsu (float): せん断終局強度Qsu
        d_rmu (float): 曲げ終局時層間変形角Rmu
        d_rmy (float): 曲げ降伏時層間変形角Rmy
    Returns:
        float: 柱のF指標
    Note:

    """

    # 変形角の定数
    d_r250 = 1 / 250.0
    d_r150 = 1 / 150.0
    d_ry = d_r150

    # 柱のF指標
    if d_qmu > d_qsu:
        # せん断柱の終局時層間変形角Rsu
        d_rsu = calc_rsu(d_qmu, d_qsu, d_rmy)

        # せん断柱のF指標
        d_cff = 1.0 + 0.27 * (d_rsu - d_r250) / (d_ry - d_r250)

        # 極脆性柱のF指標
        d_hodd = d_ho / d_dd
        if d_hodd <= 2.0:
            d_cff = 0.80

    else:
        # 曲げ柱のF指標
        if d_rmu < (d_ry):
            d_cff = 1.0 + 0.27 * (d_rmu - d_r250) / (d_ry - d_r250)
        else:
            d_myu = 2.0 * d_rmu / d_ry - 1.0
            d_myu = pow(d_myu, 0.5)
            d_cff = d_myu / (0.75 * (1.0 + 0.05 * d_rmu / d_ry))

            if d_cff > 3.2:
                d_cff = 3.2

    return d_cff


def get_result_cff(
    d_bb: float,
    d_dd: float,
    d_at: float,
    d_kei: float,
    d_pich: float,
    d_fc: float,
    d_ho: float,
    d_hh: float,
    d_sn: float,
    d_tmu: float,
    d_bmu: float,
    d_tqsu: float,
    d_bqsu: float,
) -> float:
    """柱のF指標の結果取得

    柱のF指標の結果を取得します。

    Args:
        d_bb (float): 柱幅
        d_dd (float): 柱せい
        d_at (float): 主筋断面積
        d_kei (float): 主筋径
        d_pich (float): 帯筋ピッチ
        d_fc (float): コンクリート強度
        d_ho (float): 柱内法高さ
        d_hh (float): 標準内法高さ
        d_sn (float): 軸力
        d_tmu (float): 柱頭の曲げ終局強度
        d_bmu (float): 柱脚の曲げ終局強度
        d_tqsu (float): 柱頭のせん断終局強度
        d_bqsu (float): 柱脚のせん断終局強度
    Returns:
        float: 柱のF指標
    Note:
        ・せん断終局強度は、柱頭と柱脚の平均値を採用する
        ・釣り合い軸力以上の圧縮軸力を受ける場合でも、作用軸力の曲げ終局時せん断力を採用する

    """

    # 曲げ終局強度時せん断力Qmu
    d_qmu = (d_tmu + d_bmu) / (d_ho / 1000.0)
    d_qmu = math.floor(d_qmu * 10.0) / 10.0  # 小数点第1位までを表示させる

    # せん断終局強度Qsu (平均値)
    d_qsu = (d_tqsu + d_bqsu) * 0.5
    d_qsu = math.floor(d_qsu * 10.0) / 10.0  # 小数点第1位までを表示させる

    # 柱の曲げ終局変形角の上限cRmax
    d_crmax = calc_crmax(
        d_bb, d_dd, d_at, d_kei, d_pich, d_fc, d_ho, d_sn, d_qmu, d_qsu
    )

    # 変形角の定数
    d_r250 = 1 / 250.0
    d_r30 = 1 / 30.0

    # cRmp算定におけるqの設定
    if d_pich <= 100.0:
        d_qq = 1.0
    else:
        d_qq = 1.1

    # 曲げ降伏時部材角cRmy
    # ho/D
    d_hodd = d_ho / d_dd

    # cRmy
    d_crmy = calc_crmy(d_hodd)

    if d_crmy > d_crmax:
        d_crmy = d_crmax

    # ho/Ho
    d_hohh = d_ho / d_hh

    if d_hohh > 1.0:
        d_hohh = 1.0

    # 曲げ降伏時層間変形角Rmy
    d_rmy = d_hohh * d_crmy

    if d_rmy < d_r250:
        d_rmy = d_r250

    # 曲げ塑性変形角cRmp
    d_crmp = calc_crmp(d_qmu, d_qsu, d_qq, d_crmy)

    # 曲げ終局時変形角cRmu
    d_crmu = d_crmy + d_crmp

    if d_crmu > d_crmax:
        d_crmu = d_crmax

    if d_crmu > d_r30:
        d_crmu = d_r30

    # 曲げ終局時層間変形角Rmu
    d_rmu = d_hohh * d_crmu

    if d_rmu < d_r250:
        d_rmu = d_r250

    # 柱のF指標
    d_cff = calc_cff(d_ho, d_dd, d_qmu, d_qsu, d_rmu, d_rmy)

    d_cff = math.floor(d_cff * 1000.0) / 1000.0  # 小数点第3位までを表示させる

    return d_cff


def output_result(
    str_zz: str,
    str_xx: str,
    str_yy: str,
    d_tdl2: float,
    d_tz: float,
    d_bdl2: float,
    d_bz: float,
    d_sn: float,
    d_tmu: float,
    d_bmu: float,
    d_tqsu: float,
    d_bqsu: float,
    d_ho: float,
    d_dd: float,
    d_cff: float,
) -> None:
    """計算結果出力 または メッセージ出力

    計算結果 または メッセージを出力します。┃

    Args:
        str_zz (str): 柱の位置  階名検索文字列
        str_xx (str): 柱の位置  X軸名検索文字列
        str_yy (str): 柱の位置  Y軸名検索文字列
        d_tdl2 (float): 柱頭側ピース断面(2)の柱せい
        d_tz (float): 柱頭側中立軸
        d_bdl2 (float): 柱脚側ピース断面(2)の柱せい
        d_bz (float): 柱脚側中立軸
        d_sn (float): 軸力
        d_tmu (float): 柱頭側曲げ終局強度
        d_bmu (float): 柱脚側曲げ終局強度
        d_tqsu (float): 柱頭側せん断終局強度
        d_bqsu (float): 柱脚側せん断終局強度
        d_ho (float): 柱内法高さ
        d_dd (float): 柱せい
        d_cff (float): 柱のF指標

    """

    # 計算結果 または メッセージ出力
    if (d_tdl2 < 0.0) or (d_bdl2 < 0.0):
        # 解析を中断し、メッセージ出力する
        # ピース断面分割において、柱の鉄筋重心位置を2倍した長さが柱せいの1/2を超えています。
        # 区切り線を引く
        print("－･" * 27)  # 27個を横に表示する

        # メッセージを表示する
        if d_tdl2 < 0.0:
            # 柱頭
            print("柱頭の終局強度の計算ができませんでした")

        if d_bdl2 < 0.0:
            # 柱脚
            print("柱脚の終局強度の計算ができませんでした")

        print(
            "ピース断面分割において、柱の鉄筋重心位置を2倍した長さが柱せいの1/2を超えています"
        )

        # 区切り線を引く
        print("－･" * 27)  # 27個を横に表示する

    else:
        if (d_tz < 0.0) or (d_bz < 0.0):
            # 解析を中断し、メッセージ出力する
            # ピース断面内に中立軸がありません
            # 計算に必要なパラメータがない
            # 区切り線を引く
            print("－･" * 13)  # 13個を横に表示する

            # メッセージを表示する
            if d_tz < 0.0:
                # 柱頭
                print("柱頭の曲げ終局強度Muの計算ができませんでした")

            if d_bz < 0.0:
                # 柱脚
                print("柱脚の曲げ終局強度Muの計算ができませんでした")

            # 区切り線を引く
            print("－･" * 13)  # 13個を横に表示する
        else:
            # 計算結果を出力する

            # メッセージを表示する
            print("終局強度の計算ができました")

            # 空白行を表示
            print()

            # 結果出力のタイトルを表示
            print("＜結果出力＞")

            # 区切り線を引く
            print("－･" * 17)  # 17個を横に表示する

            # フレーム名と軸名 および 階名を画面表示する
            if str_xy.upper() == "X":  # 小文字は大文字へ変換する
                # X方向フレーム
                print(str_yy + "フレ－ム ", str_xx + "軸", "（ 階:", str_zz, "）")
            else:
                # Y方向フレーム
                print(str_xx + "フレ－ム ", str_yy + "軸", "（ 階:", str_zz, "）")

            # 柱の位置情報と計算結果との間に境界線を引く
            print("- " * 26)  # 26個を横に表示する

            # 軸力を画面表示する
            print("軸力 N=", d_sn, "(kN)")

            # 破壊形式とそのＦ指標が分かるようにQmuとQsuを画面表示する
            # ここで、曲げ終局強度時せん断力Qmuを計算する
            d_qmu = (d_tmu + d_bmu) / (d_ho / 1000.0)
            d_qmu = math.floor(d_qmu * 10.0) / 10.0  # 小数点第1位までを表示させる

            # ここで、せん断終局強度の平均値を計算する
            d_qsu = (d_tqsu + d_bqsu) * 0.5
            d_qsu = math.floor(d_qsu * 10.0) / 10.0  # 小数点第1位までを表示させる

            # 曲げ終局強度と曲げ終局強度時せん断力Qmuを画面表示する
            # print("曲げ終局強度 " + str_tb + "Mu=", d_mu, "(kN･m)")
            print("曲げ終局強度   " + "TMu=", d_tmu, "(kN･m)")
            print("　　　　　　   " + "BMu=", d_bmu, "(kN･m)")
            print("　　　　　　   " + "Qmu=", d_qmu, "(kN)")

            # せん断終局強度を画面表示する
            # print("せん断終局強度 " + str_tb + "Qsu=", d_qsu, "(kN)")
            print("せん断終局強度 " + "TQsu=", d_tqsu, "(kN)")
            print("　　　　　　　 " + "BQsu=", d_bqsu, "(kN)")
            print("　　　　　　　 " + "平均=", d_qsu, "(kN)")

            # Ｆ指標と破壊形式を画面表示する
            if d_qmu > d_qsu:
                # せん断破壊
                # ho/D
                d_hodd = d_ho / d_dd
                d_hodd = (
                    math.floor(d_hodd * 1000.0) / 1000.0
                )  # 小数点第3位までを表示させる

                # ho/Dが2以下(極短柱)
                if d_hodd <= 2.0:
                    print(
                        "Ｆ指標　　　　 " + "F=",
                        d_cff,
                        "( 極脆性柱: " + "ho/D=",
                        d_hodd,
                        ")",
                    )
                else:
                    print(
                        "Ｆ指標　　　　 " + "F=",
                        d_cff,
                        "( せん断柱: " + "ho/D=",
                        d_hodd,
                        ")",
                    )
            else:
                # 曲げ破壊
                print("Ｆ指標　　　　 " + "F=", d_cff, "( 曲げ柱 )")

            # 区切り線を引く
            print("－･" * 17)  # 17個を横に表示する


def input_xyhoko() -> str:
    """解析方向の入力

    解析方向 X方向 または Y方向 の入力を行います。
    併せて、入力値がXか、Yかの判別も行います。
    X、Y以外の入力値のときは、解析方向の再入力を行います。
    (大文字と小文字の区別はしません)

    Returns:
        str: 解析方向の入力値
    Note:
        入力値は＜X方向＞、＜Y方向＞のような解析方向の出力に利用する
        -> 仕様変更で、##フレームの形式として出力させる

    """

    # 解析方向 X方向 または Y方向
    str_xy = input("解析方向 'X' または 'Y' を入力 ⇒ ")
    if str_xy.upper() == "X" or str_xy.upper() == "Y":  # 小文字は大文字へ変換する
        return str_xy
    else:
        # 再入力
        return input_xyhoko()


def input_ho() -> float:
    """内法高さの入力

    内法高さの入力を行います。
    併せて、入力値が数字の判別も行います。
    "0"と数字以外のときは、軸力の再入力を行います。

    Returns:
        float: 内法高さの入力値
    Note:
        ・入力値は文字列を整数に変換するため、数字の判別を行う

    """

    # 内法高さ
    str_ho = input("内法高さを入力（mm）⇒ ")
    try:
        d_ho = float(str_ho)

        if d_ho:
            return d_ho
        else:
            # 再入力
            return input_ho()
    except ValueError:
        # 再入力
        return input_ho()


def input_hh() -> float:
    """標準内法高さの入力

    標準内法高さの入力を行います。
    併せて、入力値が数字の判別も行います。
    数字以外のときは、軸力の再入力を行います。

    Returns:
        float: 標準内法高さの入力値
    Note:
        ・入力値は文字列を整数に変換するため、数字の判別を行う

    """

    # 標準内法高さ
    str_hh = input("標準内法高さを入力（mm）⇒ ")
    try:
        d_hh = float(str_hh)

        if d_hh:
            return d_hh
        else:
            # 再入力
            return input_hh()
    except ValueError:
        # 再入力
        return input_hh()


def input_jikuryoku() -> float:
    """軸力の入力

    軸力の入力を行います。
    併せて、入力値が数字の判別も行います。
    数字以外のときは、軸力の再入力を行います。

    Returns:
        float: 軸力の入力値
    Note:
        ・入力値は文字列を整数に変換するため、数字の判別を行う
        ・“do not use bare except”が表示されたので、except:に ValueError を追記

    """

    # 軸力
    str_sn = input("軸力を入力（kN）⇒ ")
    try:
        d_sn = float(str_sn)
        return d_sn
    except ValueError:
        # 再入力
        return input_jikuryoku()


if __name__ == "__main__":
    # 空白行を表示
    print()

    # 入力項目一覧のタイトルを表示
    print("＜入力項目一覧＞")
    # 区切り線を引く
    print("－･" * 15)  # 15個を横に表示する

    # データ入力 (全角空白は入力項目の表示調整)
    print("　1. 物件データの絶対パス")
    print("　2. 柱の位置　 階名")
    print("　3. 　〃　　  X軸名")
    print("　4. 　〃　　  Y軸名")
    print("　5. 解析方向　<X>：X方向 または <Y>：Y方向")
    print("　6. 内法高さ　　　単位：(mm)")
    print("　7. 標準内法高さ　単位：(mm)")
    print("　8. 軸力　　　　　単位：(kN)")

    # 区切り線を引く
    print("－･" * 15)  # 15個を横に表示する
    print()  # 空白行を表示

    # 計算させたい物件データの絶対パスの入力
    ss7_path = input("物件データの絶対パスを入力 ⇒ ")

    # 空白行を表示
    print()

    # 柱の位置 階名の入力
    str_zz = input("柱の位置   階名を入力 ⇒ ")

    # 柱の位置 X軸名の入力 (全角空白は入力項目の表示調整)
    str_xx = input("　〃　　  X軸名を入力 ⇒ ")

    # 柱の位置 Y軸名の入力 (全角空白は入力項目の表示調整)
    str_yy = input("　〃　　  Y軸名を入力 ⇒ ")

    # 空白行を表示
    print()

    # 解析方向の入力 X方向 または Y方向
    str_xy = input_xyhoko()

    # 空白行を表示
    print()

    # 内法高さの入力
    d_ho = input_ho()

    # 標準内法高さの入力
    d_hh = input_hh()

    # 軸力の入力
    d_sn = input_jikuryoku()

    # 空白行を表示
    print()

    resultcsv_path = ss7_path + "/Result_Shin.csv"  # 結果csvのpath
    result_name = "結果1"  # 結果名
    calculation_name = "準備計算"  # 計算項目名

    Cmd.Init()
    Cmd.Start("1.1.21.2", 1)  # 1=ログファイルをクリアする
    # データを開く
    # 1=変換して開く 3=物件データは開かない
    data = Cmd.Open(ss7_path, 1, 3)

    # データが正しく開けたことの判定C:
    if data is not None:
        # 計算結果削除
        data.DeleteResult(result_name)
        # 計算実行
        data.Calculate(result_name, calculation_name)
        # エラー情報を取得する
        err = Cmd.GetLastError()
        # 計算が正しく実行できたことの判定
        if err.IsOK() is True:
            print("結果1の計算が成功しました")
            # 計算結果データを取得する
            res = data.GetResultData(result_name)
            # 既存の結果csvを削除する
            del_csv(resultcsv_path)
            # 以下の結果を出力する/
            res.ExportResultCsv(
                "2.7.5.柱;2.5.1.コンクリート材料;2.5.3.鉄筋材料;2.5.4.鉄筋径と使用範囲",
                resultcsv_path,
                2,
                1,
                1,
            )  # 2=中断する 1=省略する 1=全部材出力
            err = Cmd.GetLastError()
            if err.IsOK() is True:
                print("結果CSVのエクスポートが成功しました")

                list_text = []  # 結果CSV出力
                list_conc = []  # コンクリート材料のデータ
                list_tekin = []  # 鉄筋材料のデータ
                list_tekinkei = []  # 鉄筋径と使用範囲のデータ
                list_colm = []  # 柱部材断面情報のデータ

                # CSV形式のファイルの読み込み
                with open(resultcsv_path, "r") as file:

                    items = csv.reader(file)

                    for item in items:
                        list_text.append(item)

                # 結果CSV出力における出力項目ごとの開始行数および終了行数を格納する
                row = search_csv(list_text)

                # 取得した出力項目ごとの開始行数および終了行数を変数へ
                i_conc_st = row[0]  # [name=コンクリート材料] の開始行数
                i_tekin_st = row[1]  # [name=鉄筋材料] の開始行数
                i_tekinkei_st = row[2]  # [name=鉄筋径と使用範囲] の開始行数
                i_colm_st = row[3]  # [name=柱部材断面情報] の開始行数
                i_conc_ed = row[4]  # [name=コンクリート材料] の終了行数
                i_tekin_ed = row[5]  # [name=鉄筋材料] の終了行数
                i_tekinkei_ed = row[6]  # [name=鉄筋径と使用範囲] の終了行数
                i_colm_ed = row[7]  # [name=柱部材断面情報] の終了行数

                # 結果CSV出力における出力項目ごとの開始行数を<data>の行数と差し替える
                # コンクリート材料のデータを読み込むための開始行数を<data>の行数と差し替える
                i_conc_st = search_csv2(list_text, i_conc_st, i_conc_ed)

                # 鉄筋材料のデータを読み込むための開始行数を<data>の行数と差し替える
                i_tekin_st = search_csv3(list_text, i_tekin_st, i_tekin_ed)

                # 鉄筋径と使用範囲のデータを読み込むための開始行数を<data>の行数と差し替える
                i_tekinkei_st = search_csv4(list_text, i_tekinkei_st, i_tekinkei_ed)

                # 柱部材断面情報のデータを読み込むための開始行数を<data>の行数と差し替える
                i_colm_st = search_csv5(list_text, i_colm_st, i_colm_ed)

                # 出力項目それぞれにリストを用意し、データを格納する
                # コンクリート材料のデータをリストに格納する
                list_conc = get_result_csv2(list_text, i_conc_st, i_conc_ed)

                # 鉄筋材料のデータを格納する
                list_tekin = get_result_csv3(list_text, i_tekin_st, i_tekin_ed)

                # 鉄筋径と使用範囲のデータを格納する
                list_tekinkei = get_result_csv4(list_text, i_tekinkei_st, i_tekinkei_ed)

                # 柱部材断面情報のデータを格納する
                list_colm = get_result_csv5(list_text, i_colm_st, i_colm_ed)

                # 柱部材断面情報のデータを取得する
                str_tb = "T"  # 柱頭
                data_c_t = get_colm(list_colm, str_zz, str_xx, str_yy, str_tb)

                str_tb = "B"  # 柱脚
                data_c_b = get_colm(list_colm, str_zz, str_xx, str_yy, str_tb)

                # メッセージ用データを初期化する
                d_tdl2: float = 0.0  # 柱頭側ピース断面分割のせい
                d_bdl2: float = 0.0  # 柱脚側ピース断面分割のせい
                d_tz: float = 0.0  # 柱頭側中立軸
                d_bz: float = 0.0  # 柱脚側中立軸
                d_cff: float = 0.0  # 柱のF指標

                # 曲げ終局強度の結果データを取得する
                ad_mu = get_mu(
                    data_c_t, data_c_b, list_conc, list_tekin, list_tekinkei, str_xy
                )

                # 取得した曲げ終局強度の結果データを出力用変数へ
                d_tdl2 = ad_mu[0]  # 柱頭側ピース断面分割のせい
                d_tz = ad_mu[1]  # 柱頭側中立軸
                d_tmu = ad_mu[2]  # 柱頭側曲げ終局強度
                d_bdl2 = ad_mu[3]  # 柱脚側ピース断面分割のせい
                d_bz = ad_mu[4]  # 柱脚側中立軸
                d_bmu = ad_mu[5]  # 柱脚側曲げ終局強度
                d_dd = ad_mu[6]  # 柱せい

                # 計算結果 または メッセージを出力する
                if (d_tdl2 >= 0) and (d_bdl2 >= 0) and (d_tz >= 0) and (d_bz >= 0):
                    # せん断終局強度の結果データを取得する
                    ad_qsu = get_qsu(
                        data_c_t,
                        data_c_b,
                        list_conc,
                        list_tekin,
                        list_tekinkei,
                        d_ho,
                        d_sn,
                        d_tmu,
                        d_bmu,
                        str_xy,
                    )

                    # 取得したせん断終局強度の結果データを出力用変数へ
                    d_tqsu = ad_qsu[0]  # 柱頭側せん断終局強度
                    d_bqsu = ad_qsu[1]  # 柱頭側せん断終局強度

                    # 柱のF指標の結果データを取得する
                    d_cff = get_cff(
                        data_c_t,
                        data_c_b,
                        list_conc,
                        list_tekinkei,
                        d_ho,
                        d_hh,
                        d_sn,
                        d_tmu,
                        d_bmu,
                        d_tqsu,
                        d_bqsu,
                        str_xy,
                    )

                    # 計算結果を出力します
                    output_result(
                        str_zz,
                        str_xx,
                        str_yy,
                        d_tdl2,
                        d_tz,
                        d_bdl2,
                        d_bz,
                        d_sn,
                        d_tmu,
                        d_bmu,
                        d_tqsu,
                        d_bqsu,
                        d_ho,
                        d_dd,
                        d_cff,
                    )
                else:
                    d_tmu = 0.0
                    d_bmu = 0.0
                    d_tqsu = -1.0
                    d_bqsu = -1.0
                    output_result(
                        str_zz,
                        str_xx,
                        str_yy,
                        d_tdl2,
                        d_tz,
                        d_bdl2,
                        d_bz,
                        d_sn,
                        d_tmu,
                        d_bmu,
                        d_tqsu,
                        d_bqsu,
                        d_ho,
                        d_dd,
                        d_cff,
                    )
            else:
                print("結果CSVのエクスポートが失敗しました")
                # データを閉じる
                data.Close(1)  # 1=保存する
        else:
            print("計算が失敗しました")
            # データを閉じる
            data.Close(2)  # 2=保存しない
    else:
        print("データが開けませんでした")

    Cmd.End(1)  # 1＝保存して終了
