"""杭の許容支持力の一覧表

杭の許容支持力において、設計指針、工法すべての組み合わせを計算し、一覧表にまとめます。
また、設計指針、工法に応じた先端N値の採用範囲を自動認識し、許容支持力を算出します。

Copyright (C) 2026 UNION SYSTEM Inc.

"""


from Python import Ss7Python as Cmd
import os
import shutil
import csv
from openpyxl import Workbook
from openpyxl import load_workbook


def import_csv(file_path: str) -> list:
    """CSVの読み込み

    指定したCSVファイルを読み込み、
    CSVデータをリストに格納する。

    Args:
        file_path (str): CSVファイルのパス
    Returns:
        list: CSVデータのリスト

    """

    data: list = []  # CSVデータのリスト
    # CSVファイルを読み込む
    with open(file_path, mode="r", encoding="cp932") as file:
        reader = csv.reader(file)
        for row in reader:
            data.append(row)
    return data


def get_row_csv(data: list, target: str) -> list:
    """指定した文字列のCSVの行番号の取得

    指定した文字列のCSVファイルの行番号を取得する。

    Args:
        data (list): 文字列のリスト
        target (str): 検索したい文字列
    Returns:
        list: 行番号

    """

    line_no: list = []  # 行番号
    line_no = [i for i, line_s in enumerate(data) if target in line_s]
    return line_no


def export_csv(file_path: str, csvdata: list) -> None:
    """CSVの書き込み

    指定したCSVファイルへデータを書き込む。

    Args:
        file_path (str): CSVファイルのパス
        csvdata (list): データ

    """

    # CSVファイルへ書き込む
    with open(file_path, mode="w") as file:
        writer = csv.writer(file, lineterminator="\n")
        writer.writerows(csvdata)


class RAClass:
    """杭の許容支持力の計算クラス

    杭の許容支持力を計算するクラスです。

    Attributes:
        m_ss7_version（str）:『SS7』のバージョン
        m_ss7_path（str）:『SS7』データのパス
        m_kiso（int）:基礎形式
        m_formula_num（int）:設計指針の数
        m_method_num（int）:工法の数
        m_formula（int）:設計指針
        m_method（int）:工法
        m_member_num（int）:代表部材の数
        m_floor（list）:層
        m_xjiku（list）:X軸
        m_yjiku（list）:Y軸
        m_pfugo（list）:杭符号
        m_ra（list）:許容支持力

    """

    def __init__(self, ss7_version: str, ss7_path: str) -> None:
        self.m_ss7_version: str = ss7_version
        self.m_ss7_path: str = ss7_path
        self.m_kiso: int = 0
        self.m_formula_num: int = 0
        self.m_method_num: int = 0
        self.m_formula: int = 0
        self.m_method: int = 0
        self.m_member_num: int = 0
        self.m_floor: list = []
        self.m_xjiku: list = []
        self.m_yjiku: list = []
        self.m_pfugo: list = []
        self.m_ra: list = []

    def cal_ra(self) -> None:
        """許容支持力の計算

        入力CSVの[杭の支持力計算]で設計指針、工法、先端N値の採用範囲を登録し、
        『SS7』で許容支持力を計算する。
        結果CSVの「8.9.1.杭」より設計Raを取得する。

        """

        # 入力CSVの出力
        self.inp_ss7datacsv()

        # 新たな入力CSVの作成
        self.make_inputcsv_new()

        # CSV新規作成
        self.create_datacsv()

        # 結果CSVの出力
        self.out_ss7datacsv()

        # 許容Raの取得
        self.get_ss7datacsv_ra()

    def inp_ss7datacsv(self) -> None:
        """入力CSVの出力

        『SS7』を起動して入力CSVを出力する。

        """

        # 『SS7』入力CSVを出力する
        Cmd.Init()
        Cmd.Start(self.m_ss7_version, 1)  # 1=ログファイルをクリアする

        # データを開く
        data = Cmd.Open(self.m_ss7_path, 1, 3)  # 1=変換して開く 3=物件データは開かない

        # データが正しく開けたことの判定
        if data is not None:

            # 入力データを取得する
            inp = data.GetInputData()

            # 入力CSVをエクスポートする
            inp.ExportInputCsv(self.m_ss7_path + r"\input.csv", 1, 1)
            err = Cmd.GetLastError()

            # 入力CSVのエクスポートに失敗した場合はメッセージを出力し、プログラムを終了する
            if err.IsOK() is False:
                print("入力CSVのエクスポートに失敗しました")
                # データを閉じる
                data.Close(2)  # 2=保存しない
            else:
                print("入力CSVのエクスポートに成功しました")

            # 入力CSVを読み込む
            self.m_inpcsvdata = import_csv(self.m_ss7_path + r"\input.csv")

        # データを閉じる
        data.Close(2)  # 2=保存しない
        Cmd.End(1)  # 1=保存して終了

    def make_inputcsv_new(self) -> None:
        """新たな入力CSVの作成

        入力CSV[杭の支持力計算]で設計指針、工法、先端N値の採用範囲を登録し、新たな入力CSVを作成する。

        """

        # 入力CSVの[杭の支持力計算]の設計指針を変更
        row_formula: list = get_row_csv(self.m_inpcsvdata, "設計指針")
        row_def: int = row_formula[0]
        self.m_inpcsvdata[row_def][6] = self.m_formula

        # 入力CSVの[杭の支持力計算]の工法を変更
        row_method: list = get_row_csv(self.m_inpcsvdata, "工法")
        row_def = row_method[0]
        self.m_inpcsvdata[row_def][6] = self.m_method

        # 入力CSVの[杭の支持力計算]の先端Nの採用範囲を変更
        row_range: list = get_row_csv(self.m_inpcsvdata, "採用範囲")
        row_def = row_range[0]
        if 3 == self.m_kiso:  # 既製杭
            if 1 == self.m_formula:  # 告示
                self.m_inpcsvdata[row_def][6] = 1  # 1D4D
            else:  # 告示以外
                if 1 == self.m_method:  # 打込
                    self.m_inpcsvdata[row_def][6] = 1
                else:  # 打込以外
                    self.m_inpcsvdata[row_def][6] = 2  # 1D1D
        else:  # 場所打ち杭
            if 1 == self.m_formula:  # 告示
                self.m_inpcsvdata[row_def][6] = 1
            else:  # 告示以外
                self.m_inpcsvdata[row_def][6] = 2

        # 新たな入力CSVの作成
        export_csv(self.m_ss7_path + r"\input_after.csv", self.m_inpcsvdata)

    def create_datacsv(self) -> None:
        """『SS7』CSV新規作成

        新たに作成した『SS7』入力CSVデータから『SS7』データを新規作成する。

        """

        # 『SS7』CSV新規作成
        Cmd.Init()
        Cmd.Start(self.m_ss7_version, 1)  # 1=ログファイルをクリアする

        # 新たに作成した入力CSVデータから『SS7』データを新規作成する
        Cmd.CreateDataCsv(
            self.m_ss7_path + r"\input_after.csv",
            self.m_ss7_path + r"\input_after.ikn",
            1,
        )  # 1=上書きする

        # エラー情報を取得する
        err = Cmd.GetLastError()

        # CSV新規作成できたことの判定
        if err.IsOK() is True:
            print("CSV新規作成が成功しました")
        else:
            print("CSV新規作成が失敗しました")

        Cmd.End(1)  # 1=保存して終了

    def out_ss7datacsv(self) -> None:
        """結果CSVデータの出力

        『SS7』を起動して結果CSVを出力する。

        """

        # 『SS7』結果CSVを出力する
        Cmd.Init()
        Cmd.Start(self.m_ss7_version, 1)  # 1=ログファイルをクリアする

        # データを開く
        data = Cmd.Open(
            self.m_ss7_path + r"\input_after.ikn", 1, 3
        )  # 1=変換して開く 3=物件データは開かない

        # データが正しく開けたことの判定
        if data is not None:
            # 計算実行
            data.DeleteResult("結果1")  # 結果1がある場合削除
            data.Calculate("結果1", "断面算定")
            # エラー情報を取得する
            err = Cmd.GetLastError()

            # 計算が正しく実行できたことの判定
            if err.IsOK() is True:
                print("計算が成功しました")

                # 計算結果データを取得する
                res = data.GetResultData("結果1")

                # 結果CSVをエクスポートする
                res.ExportResultCsv(
                    "8.9.1.杭",
                    self.m_ss7_path + r"\output.csv",
                    1,
                    1,
                    2,
                )
                err = Cmd.GetLastError()

                # 結果CSVのエクスポートに失敗した場合はメッセージを出力し、プログラムを終了する
                if err.IsOK() is False:
                    print("結果CSVのエクスポートに失敗しました")
                    # データを閉じる
                    data.Close(2)  # 2=保存しない
                else:
                    print("結果CSVのエクスポートに成功しました")

                # 結果CSVを読み込む
                self.m_outcsvdata = import_csv(
                    self.m_ss7_path + r"\output.csv"
                )
            else:
                print("計算が失敗しました")
                # データを閉じる
                data.Close(2)  # 2=保存しない

        # データを閉じる
        data.Close(2)  # 2=保存しない
        Cmd.End(1)  # 1=保存して終了

    def get_ss7datacsv_ra(self) -> None:
        """許容支持力の取得

        結果CSV「8.9.1.杭」から設計Raを取得する。

        """
        row: int = 0
        while True:
            if 2 * row + 13 == len(self.m_outcsvdata):
                break
            self.m_floor.append([])
            self.m_xjiku.append([])
            self.m_yjiku.append([])
            self.m_pfugo.append([])
            self.m_ra.append([])
            self.m_floor[row].append(self.m_outcsvdata[2 * row + 13][0])
            self.m_xjiku[row].append(self.m_outcsvdata[2 * row + 13][1])
            self.m_yjiku[row].append(self.m_outcsvdata[2 * row + 13][2])
            self.m_pfugo[row].append(self.m_outcsvdata[2 * row + 13][3])
            self.m_ra[row].append(self.m_outcsvdata[2 * row + 13][13])
            row += 1
            self.member_num = row

    def out_ra(self) -> None:
        """許容支持力結果の出力

        許容支持力結果を出力する。

        """

        # ワークブックの新規作成と保存
        wb = Workbook()

        # Excelファイルを作成する
        wb.save(self.m_ss7_path + r"\out_ra.xlsx")

        # ワークブックを開く
        wb = load_workbook(self.m_ss7_path + r"\out_ra.xlsx")

        # デフォルトの'Sheet'を削除
        for ws in wb.worksheets:
            if ws.title == "Sheet":
                del wb["Sheet"]

        # ワークシートの作成
        wb.create_sheet("許容Ra")

        # ワークシートの選択
        ws = wb["許容Ra"]
        wb.active = ws

        # セルに許容Raを書き込む

        # ヘッダー
        if 3 == self.m_kiso:  # 既製杭
            ws.cell(row=1, column=5, value="告示")
            ws.cell(row=1, column=9, value="学会2001")
            ws.cell(row=1, column=13, value="学会2019")
            ws.cell(row=1, column=18, value="東京")
            ws.cell(row=1, column=22, value="神奈川")
            ws.cell(row=1, column=26, value="大阪")
            ws.cell(row=2, column=1, value="層")
            ws.cell(row=2, column=2, value="X軸")
            ws.cell(row=2, column=3, value="Y軸")
            ws.cell(row=2, column=4, value="杭符号")
            ws.cell(row=2, column=5, value="打込")
            ws.cell(row=2, column=6, value="埋込(セメ)")
            ws.cell(row=2, column=7, value="埋込(認定)")
            ws.cell(row=2, column=8, value="摩擦(認定)")
            ws.cell(row=2, column=9, value="打込")
            ws.cell(row=2, column=10, value="埋込(杭周)")
            ws.cell(row=2, column=11, value="埋込(認定)")
            ws.cell(row=2, column=12, value="摩擦(認定)")
            ws.cell(row=2, column=13, value="打込")
            ws.cell(row=2, column=14, value="埋込(プレ)")
            ws.cell(row=2, column=15, value="埋込(認定)")
            ws.cell(row=2, column=16, value="摩擦(認定)")
            ws.cell(row=2, column=17, value="埋込(中堀)")
            ws.cell(row=2, column=18, value="打込")
            ws.cell(row=2, column=19, value="埋込(セメ)")
            ws.cell(row=2, column=20, value="埋込(認定)")
            ws.cell(row=2, column=21, value="摩擦(認定)")
            ws.cell(row=2, column=22, value="打込")
            ws.cell(row=2, column=23, value="埋込(セメ)")
            ws.cell(row=2, column=24, value="埋込(認定)")
            ws.cell(row=2, column=25, value="摩擦(認定)")
            ws.cell(row=2, column=26, value="打込")
            ws.cell(row=2, column=27, value="埋込(セメ)")
            ws.cell(row=2, column=28, value="埋込(認定)")
            ws.cell(row=2, column=29, value="摩擦(認定)")
        else:  # 場所打ち杭
            ws.cell(row=1, column=5, value="告示")
            ws.cell(row=1, column=7, value="学会2001")
            ws.cell(row=1, column=9, value="学会2019")
            ws.cell(row=1, column=11, value="東京A")
            ws.cell(row=1, column=13, value="東京B")
            ws.cell(row=1, column=15, value="神奈川")
            ws.cell(row=1, column=17, value="大阪")
            ws.cell(row=2, column=1, value="層")
            ws.cell(row=2, column=2, value="X軸")
            ws.cell(row=2, column=3, value="Y軸")
            ws.cell(row=2, column=4, value="杭符号")
            ws.cell(row=2, column=5, value="ベノト")
            ws.cell(row=2, column=6, value="その他")
            ws.cell(row=2, column=7, value="ベノト")
            ws.cell(row=2, column=8, value="その他")
            ws.cell(row=2, column=9, value="ベノト")
            ws.cell(row=2, column=10, value="その他")
            ws.cell(row=2, column=11, value="ベノト")
            ws.cell(row=2, column=12, value="その他")
            ws.cell(row=2, column=13, value="ベノト")
            ws.cell(row=2, column=14, value="その他")
            ws.cell(row=2, column=15, value="ベノト")
            ws.cell(row=2, column=16, value="その他")
            ws.cell(row=2, column=17, value="ベノト")
            ws.cell(row=2, column=18, value="その他")

        for i in range(0, self.member_num, 1):
            # 層
            ws.cell(row=i + 3, column=1, value=self.m_floor[i][0])
            # X軸
            ws.cell(row=i + 3, column=2, value=self.m_xjiku[i][0])
            # Y軸
            ws.cell(row=i + 3, column=3, value=self.m_yjiku[i][0])
            # 杭符号
            ws.cell(row=i + 3, column=4, value=self.m_pfugo[i][0])

            for j in range(0, len(self.m_ra[0]), 1):
                # 許容Ra
                ws.cell(row=i + 3, column=j + 5, value=float(self.m_ra[i][j]))

        # ワークブックの保存
        wb.save(self.m_ss7_path + r"\out_ra.xlsx")


def main(ss7_version: str, ss7_path: str) -> None:
    """メイン処理

    最初に動く関数です。

    Args:
        ss7_version (str): 『SS7』のバージョン
        ss7_path (str): 『SS7』データのパス

    """

    # 許容支持力計算クラスのオブジェクトの作成
    ra = RAClass(ss7_version, ss7_path)

    # 基礎形式の取得
    ra.inp_ss7datacsv()
    row_kiso: list = get_row_csv(ra.m_inpcsvdata, "基礎形式")
    ra.m_kiso = int(ra.m_inpcsvdata[row_kiso[0]][4])

    # 設計指針の数をセット
    if 3 == ra.m_kiso:  # 既製杭
        ra.m_formula_num = 6
    elif 2 == ra.m_kiso:  # 場所打ち杭
        ra.m_formula_num = 7
    else:
        print("杭基礎でない")

    # 許容支持力の計算
    icount: int = 1
    for i in range(ra.m_formula_num):  # 設計指針
        ra.m_formula = i + 1

        # 工法の数をセット
        if 3 == ra.m_kiso:  # 既製杭
            if 3 == ra.m_formula:  # 基礎指針2019
                ra.m_method_num = 5
            else:
                ra.m_method_num = 4
        else:  # 場所打ち杭
            ra.m_method_num = 2

        for j in range(ra.m_method_num):  # 工法
            print(icount)
            ra.m_method = j + 1
            # 許容支持力の計算
            ra.cal_ra()
            icount += 1

    # 許容支持力結果の出力
    print("出力")
    ra.out_ra()

    # 中間ファイルの削除
    os.remove(ss7_path + r"\input.csv")
    os.remove(ss7_path + r"\input_after.csv")
    os.remove(ss7_path + r"\output.csv")
    shutil.rmtree(ss7_path + r"\input_after.ikn")


if __name__ == "__main__":

    # 環境設定の指定

    # 『SS7』のバージョン
    ss7_version: str = "1.1.21.2"

    # 『SS7』データのパス
    ss7_path: str = r"C:\UsrData\Ss7Data\Ra_test.ikn"

    main(ss7_version, ss7_path)
